package gojiadapter

import (
	"bytes"
	"context"
	"encoding/base64"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"

	"github.com/aws/aws-lambda-go/events"
	goji "goji.io"
)

type ApiGatewayHandler = func(ctx context.Context, req events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error)

type GojiGatewayMux struct {
	Mux     *goji.Mux
	Handler ApiGatewayHandler
}

type shimResponseWriter struct {
	bodyBuffer *bytes.Buffer
	header     http.Header
	statusCode *int
}

func (rw *shimResponseWriter) Header() http.Header {
	return rw.header
}

func (rw *shimResponseWriter) Write(payload []byte) (int, error) {
	return rw.bodyBuffer.Write(payload)
}

func (rw *shimResponseWriter) WriteHeader(statusCode int) {
	code := statusCode
	rw.statusCode = &code
}

func RequestFromGatewayRequest(ctx context.Context, req events.APIGatewayProxyRequest) *http.Request {
	// body to reader
	bodyBuffer := bytes.NewBufferString(req.Body)
	var bodyReader io.ReadCloser = ioutil.NopCloser(bodyBuffer)
	if req.IsBase64Encoded {
		bodyReader = ioutil.NopCloser(base64.NewDecoder(base64.StdEncoding, bodyBuffer))
	}

	queryParams := url.Values{}
	for key, val := range req.QueryStringParameters {
		queryParams.Add(key, val)
	}
	// url reconstruction
	reqUrl := &url.URL{
		Scheme:   "http",
		Host:     "shim",
		Path:     req.Path,
		RawQuery: queryParams.Encode(),
	}

	headers := http.Header{}
	// Of course headers are different
	for header, headerVal := range req.Headers {
		headers.Set(header, headerVal)
	}

	request := &http.Request{
		Method:        req.HTTPMethod,
		URL:           reqUrl,
		Header:        headers,
		Proto:         "HTTP/1.1",
		Body:          bodyReader,
		ContentLength: int64(bodyBuffer.Len()),
	}
	return request.WithContext(ctx)
}

func NewMux() *GojiGatewayMux {
	mux := goji.NewMux()

	handler := func(ctx context.Context, req events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
		// req to http.Request
		request := RequestFromGatewayRequest(ctx, req)
		// create shim http.ResponseWriter
		respWriter := &shimResponseWriter{
			bodyBuffer: &bytes.Buffer{},
			header:     http.Header{},
			statusCode: nil,
		}

		mux.ServeHTTP(respWriter, request)

		statusCode := http.StatusInternalServerError
		if respWriter.statusCode != nil || respWriter.bodyBuffer.Len() > 0 {
			statusCode = http.StatusOK
		}

		headers := map[string]string{}
		for hkey, hval := range respWriter.Header() {
			if len(hval) > 0 {
				headers[hkey] = hval[0]
			}
		}

		// respWriter back to response
		return events.APIGatewayProxyResponse{
			StatusCode:      statusCode,
			Headers:         headers,
			Body:            respWriter.bodyBuffer.String(),
			IsBase64Encoded: false,
		}, nil
	}

	return &GojiGatewayMux{
		Mux:     mux,
		Handler: handler,
	}
}
