package kraken

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"time"
)

const (
	featuredURLTemplate    = "https://%s/kraken/streams/featured?limit=%d"
	gameURLTemplate        = "https://%s/kraken/streams"
	accessTokenURLTemplate = "https://%s/channels/%s/access_token"
)

// Client is a client for making requests to Kraken
type Client struct {
	HttpClient  *http.Client
	ApiHostname string
	ClientID    string
}

// AccessToken is the token response from usher for requesting playlists
type AccessToken struct {
	Token string `json:"token"`
	Sig   string `json:"sig"`
}

type krakenStream struct {
	ID        int64         `json:"_id"`
	Viewers   int           `json:"viewers"`
	Channel   krakenChannel `json:"channel"`
	CreatedAt time.Time     `json:"created_at"`
}

type krakenStreamsResponse struct {
	Streams []krakenStream `json:"streams"`
}

type krakenFeaturedResponse struct {
	Featured []krakenFeatured `json:"featured"`
}

type krakenFeatured struct {
	Stream krakenStream `json:"stream"`
	Title  string       `json:"title"`
}

type krakenChannel struct {
	ID          int64  `json:"_id"`
	Mature      bool   `json:"mature"`
	DisplayName string `json:"display_name"`
	Name        string `json:"name"`
	Game        string `json:"game"`
}

// Stream is a collection of info needed to show a stream on the wall
type Stream struct {
	ID          int64  `json:"id"`
	Channel     string `json:"channel_name"`
	DisplayName string `json:"display_name"`
	Viewers     int    `json:"viewers"`
	Game        string `json:"game"`
	Mature      bool   `json:"mature"`

	CreatedAt time.Time `json:"created_at"`
}

// GetFeaturedStreams returns a collection of streams that are Featured
func (k *Client) GetFeaturedStreams(ctx context.Context, limit int) ([]*Stream, error) {
	url := fmt.Sprintf(featuredURLTemplate, k.ApiHostname, limit)
	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil {
		return nil, err
	}

	req.Header.Add("Accept", "application/vnd.twitchtv.v5+json")
	req.Header.Add("Client-ID", k.ClientID)

	resp, err := k.HttpClient.Do(req)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("Got unexpected status code from kraken: %s", resp.Status)
	}

	var featuredResponse krakenFeaturedResponse
	err = json.NewDecoder(resp.Body).Decode(&featuredResponse)
	if err != nil {
		return nil, err
	}

	streams := make([]*Stream, 0, len(featuredResponse.Featured))
	for _, krakenFeatured := range featuredResponse.Featured {
		stream := &Stream{
			ID:          krakenFeatured.Stream.ID,
			Channel:     krakenFeatured.Stream.Channel.Name,
			DisplayName: krakenFeatured.Stream.Channel.DisplayName,
			Viewers:     krakenFeatured.Stream.Viewers,
			Game:        krakenFeatured.Stream.Channel.Game,
			Mature:      krakenFeatured.Stream.Channel.Mature,
		}
		streams = append(streams, stream)
	}

	return streams, nil
}

// GetStreamsByGame returns a collection of streams that are in the specified game
func (k *Client) GetStreamsByGame(ctx context.Context, game, language string, limit int) ([]*Stream, error) {
	url := fmt.Sprintf(gameURLTemplate, k.ApiHostname)
	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil {
		return nil, err
	}

	q := req.URL.Query()
	q.Add("game", game)
	q.Add("limit", strconv.Itoa(limit))

	if language != "" {
		q.Add("langauge", "en")
	}
	req.URL.RawQuery = q.Encode()

	req.Header.Add("Accept", "application/vnd.twitchtv.v5+json")
	req.Header.Add("Client-ID", k.ClientID)

	resp, err := k.HttpClient.Do(req)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("Got unexpected status code from kraken: %s", resp.Status)
	}

	var streamsResponse krakenStreamsResponse
	err = json.NewDecoder(resp.Body).Decode(&streamsResponse)

	if err != nil {
		return nil, err
	}

	streams := make([]*Stream, 0, len(streamsResponse.Streams))
	for _, krakenStream := range streamsResponse.Streams {
		stream := &Stream{
			ID:          krakenStream.ID,
			Channel:     krakenStream.Channel.Name,
			DisplayName: krakenStream.Channel.DisplayName,
			Viewers:     krakenStream.Viewers,
			Game:        krakenStream.Channel.Game,
			Mature:      krakenStream.Channel.Mature,
			CreatedAt:   krakenStream.CreatedAt,
		}
		streams = append(streams, stream)
	}

	return streams, nil
}

// GetAccessToken makes a request to Kraken on behalf of the user to get an accesstoken
func (k *Client) GetAccessToken(ctx context.Context, streamName string) (*AccessToken, error) {
	url := fmt.Sprintf(accessTokenURLTemplate, k.ApiHostname, streamName)

	req, err := http.NewRequest(http.MethodGet, url, nil)
	if err != nil {
		return nil, err
	}

	req.Header.Add("Accept", "application/vnd.twitchtv.v5+json")
	req.Header.Add("Client-ID", k.ClientID)
	// TODO: make this a part of the client struct instead of hardcoding it here
	req.Header.Add("X-Device-Id", "twitch-web-wall-mason")

	resp, err := k.HttpClient.Do(req)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("Got unexpected status code from kraken: %s", resp.Status)
	}

	var token AccessToken
	err = json.NewDecoder(resp.Body).Decode(&token)
	if err != nil {
		return nil, err
	}

	return &token, nil
}
