package main

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"log"
	"math/rand"
	"net/http"
	"net/url"
	"os"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/web/wall/cmd/mason/internal/gojiadapter"
	"code.justin.tv/web/wall/cmd/mason/internal/kraken"
	"github.com/aws/aws-lambda-go/lambda"
	"goji.io"
	"goji.io/pat"
)

const playlistTemplate = "https://usher.ttvnw.net/api/channel/hls/%s.m3u8?%s"

var krakenClient *kraken.Client

/*
	API:
	/config
		General config information for the wall

	/streams
		List of streams that can be used to show in the wall

	/playlist/stream
		Gets the url for a specific stream to be watched
*/

func mustGetEnv(key string) string {
	res := os.Getenv(key)
	if res == "" {
		log.Fatalf("Env var %s must be set", key)
	}
	return res
}

func init() {
	krakenClient = &kraken.Client{
		HttpClient:  &http.Client{},
		ApiHostname: mustGetEnv("KRAKEN_HOSTNAME"),
		ClientID:    mustGetEnv("KRAKEN_CLIENT_ID"),
	}
	rand.Seed(time.Now().UnixNano())
}

type playlistResponse struct {
	URL string `json:"url"`
}

func configHandler(w http.ResponseWriter, r *http.Request) {
	w.WriteHeader(http.StatusNotImplemented)
}

func streamsHandler(w http.ResponseWriter, r *http.Request) {
	limit := 100
	queryParams := r.URL.Query()
	if limitParam, ok := queryParams["limit"]; ok {
		i, err := strconv.Atoi(limitParam[0])
		if err == nil {
			limit = i
		}
	}

	var streams []*kraken.Stream

	ctx := r.Context()

	language := queryParams["language"]
	since := queryParams["since"]

	if gamesStr, ok := queryParams["games"]; ok {
		games := strings.Split(gamesStr[0], ",")
		for _, game := range games {
			gameStreams, err := krakenClient.GetStreamsByGame(ctx, game, language, limit)
			if err != nil {
				w.WriteHeader(http.StatusInternalServerError)
				return
			}

			var minLim time.Time

			if since != "" {
				secs, err := strconv.Atoi(since)
				if err != nil {
					w.WriteHeader(http.StatusInternalServerError)
					return
				}
				minLim = time.Now().Add(-1 * since * time.Second)

				for _, stream := range gameStreams {
					if stream.CreatedAt.After(minLim) {
						streams = append(streams, stream)
					}
				}
			} else {
				streams = append(streams, gameStreams...)
			}
		}

		rand.Shuffle(len(streams), func(i, j int) { streams[i], streams[j] = streams[j], streams[i] })

		l := min(limit, len(streams))
		streams = streams[:l]
	} else {
		var err error

		streams, err = krakenClient.GetFeaturedStreams(ctx, limit)
		if err != nil {
			w.WriteHeader(http.StatusInternalServerError)
			return
		}
	}

	w.WriteHeader(http.StatusOK)
	if err := json.NewEncoder(w).Encode(streams); err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		return
	}
}

func getPlaylistURL(ctx context.Context, channel string) (string, error) {
	token, err := krakenClient.GetAccessToken(ctx, channel)
	if err != nil {
		return "", err
	}

	// add acecess token to url
	v := url.Values{}
	v.Add("player", "twitchweb")
	v.Add("allow_source", "true")
	v.Add("p", "1234567")
	v.Add("token", token.Token)
	v.Add("sig", token.Sig)

	return fmt.Sprintf(playlistTemplate, channel, v.Encode()), nil
}

func playlistHandler(w http.ResponseWriter, r *http.Request) {
	channel := pat.Param(r, "channel")

	playlistURL, err := getPlaylistURL(r.Context(), channel)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	resp := playlistResponse{
		URL: playlistURL,
	}
	err = json.NewEncoder(w).Encode(&resp)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		return
	}
	w.WriteHeader(http.StatusOK)
}

func playlistProxyHandler(w http.ResponseWriter, r *http.Request) {
	channel := pat.Param(r, "channel")

	playlistURL, err := getPlaylistURL(r.Context(), channel)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	resp, err := http.Get(playlistURL)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	if resp.StatusCode != http.StatusOK {
		w.WriteHeader(resp.StatusCode)
		return
	}

	w.WriteHeader(http.StatusOK)
	for header, vals := range resp.Header {
		for _, val := range vals {
			w.Header().Add(header, val)
		}
	}
	io.Copy(w, resp.Body)
}

func corsWrapper(handler http.HandlerFunc) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		w.Header().Add("Access-Control-Allow-Origin", "*")
		handler(w, r)
	}
}

func setupMux(mux *goji.Mux) *goji.Mux {
	mux.HandleFunc(pat.Get("/config"), corsWrapper(configHandler))
	mux.HandleFunc(pat.Get("/streams"), corsWrapper(streamsHandler))
	mux.HandleFunc(pat.Get("/playlist/:channel"), corsWrapper(playlistHandler))
	mux.HandleFunc(pat.Get("/watch/:channel.m3u8"), corsWrapper(playlistProxyHandler))

	return mux
}

func main() {

	if os.Getenv("_LAMBDA_SERVER_PORT") != "" {
		log.Println("_LAMBDA_SERVER_PORT variable is not set -- running in local development mode on port 8000")
		mux := setupMux(goji.NewMux())
		http.ListenAndServe("localhost:8000", mux)
	} else {
		log.Println("_LAMBDA_SERVER_PORT variable is set -- running in API Gateway compatibility mode")
		gatewayMux := gojiadapter.NewMux()
		setupMux(gatewayMux.Mux)
		lambda.Start(gatewayMux.Handler)
	}
}

func min(x, y int) int {
	if x > y {
		return y
	}
	return x
}
