import * as React from 'react';

import { shuffle } from 'wall/lib/shuffle';
import {
  Stream,
  getHLSURLForChannel,
  getFeaturedChannels,
} from 'wall/lib/kraken';
import { Mosaic } from 'wall/components/mosaic';
import { VideoFrame } from 'wall/components/video-frame';
import { SITE_REFRESH_RATE, POOL_REFRESH_RATE } from 'config';
import { shouldUseNativeVideo } from 'wall/lib/support';

import './style.css';

interface Props {}

interface Streamable extends Stream {
  hlsURL: string;
}

interface State {
  activeChannels: Array<Streamable>;
  height: number;
  width: number;
  suppressNotice: boolean;
}

export class App extends React.PureComponent<Props, State> {
  private refreshPoolTimer: number = -1;
  private channelPool: Array<Stream> = [];

  constructor(props: Props) {
    super(props);

    this.state = {
      height: window.innerHeight,
      width: window.innerWidth,
      activeChannels: [],
      suppressNotice: false,
    };

    setTimeout(() => {
      this.setState({ suppressNotice: true });
    }, 10000);

    this.getChannels();
  }

  public render() {
    if (this.state.activeChannels.length === 0) {
      return <div>Loading...</div>;
    }
    if (!shouldUseNativeVideo && !this.state.suppressNotice) {
      return (
        <div className="notice">
          This browser will not perform as well as we would like to showcase
          Twitch. <b>Please use Safari on Mac or Edge on Windows</b>, if this is
          does not work for your use case please ask in #wall
          <br />
          <br />
          Streams will be loaded regardless in 10 seconds.
        </div>
      );
    }
    let vids = this.state.activeChannels.map((channel, idx) => (
      <div
        key={channel.channel}
        className={`vid${idx + 1} video-box`}
        onClick={this.toggleFullscreen}
      >
        <VideoFrame
          name={channel.displayName}
          channel={channel.channel}
          viewers={channel.viewers}
          game={channel.game}
          hlsURL={channel.hlsURL}
          replaceChannel={this.replaceChannel}
        />
      </div>
    ));
    return (
      <Mosaic
        height={this.state.height}
        width={this.state.width}
        onReplace={this.replaceIndex}
      >
        {vids}
      </Mosaic>
    );
  }

  public componentDidMount() {
    window.addEventListener('resize', this.handleResize);
    setTimeout(() => {
      // May god have mercy on my soul...
      window.location.reload(true);
    }, SITE_REFRESH_RATE);
  }

  public componentWillUnmount() {
    window.removeEventListener('resize', this.handleResize);
    clearInterval(this.refreshPoolTimer);
  }

  private async getChannels() {
    let featuredChannels = await getFeaturedChannels();
    featuredChannels = featuredChannels.filter(resp => !resp.mature);

    shuffle(featuredChannels);
    let topChannels = featuredChannels.slice(0, 9);
    this.channelPool = featuredChannels.slice(10);

    let streamableChannels = await Promise.all(
      topChannels.map(async stream => {
        let hlsURL = await getHLSURLForChannel(stream.channel);
        return {
          ...stream,
          hlsURL,
        };
      }),
    );

    this.setState({
      activeChannels: streamableChannels,
    });

    this.refreshPoolTimer = setInterval(() => {
      this.refreshPool();
    }, POOL_REFRESH_RATE);
  }

  private async refreshPool() {
    let featuredChannels = await getFeaturedChannels();

    featuredChannels = featuredChannels.filter(stream => {
      for (let i = 0; i < this.state.activeChannels.length; i++) {
        if (this.state.activeChannels[i].id === stream.id) {
          return false;
        }
      }
      return true;
    });

    this.channelPool = shuffle(featuredChannels);
  }

  private replaceIndex = async (idx: number) => {
    let candidate = this.channelPool.shift();
    if (candidate) {
      let hlsURL = await getHLSURLForChannel(candidate.channel);
      let newActive = this.state.activeChannels.slice();
      this.channelPool.push(this.state.activeChannels[idx]);
      newActive[idx] = {
        ...candidate,
        hlsURL,
      };

      this.setState({
        activeChannels: newActive,
      });
    }
  };

  private replaceChannel = (channel: string) => {
    for (let i = 0; i < this.state.activeChannels.length; i++) {
      if (this.state.activeChannels[i].channel === channel) {
        return this.replaceIndex(i);
      }
    }
  };

  private toggleFullscreen = () => {
    if (!document.fullscreenElement) {
      if (document.documentElement.requestFullscreen) {
        document.documentElement.requestFullscreen();
      }
      let el = document.documentElement as any;
      if (el.webkitRequestFullScreen) {
        el.webkitRequestFullScreen();
      }
      if (el.mozRequestFullScreen) {
        el.mozRequestFullScreen();
      }
      if (el.msRequestFullscreen) {
        el.msReuqestFullscreen();
      }
    }
  };

  private handleResize = () => {
    this.setState({
      height: window.innerHeight,
      width: window.innerWidth,
    });
  };
}
