import * as React from 'react';
import * as Hls from 'hls.js';
import './style.css';

interface Props {
  hlsURL: string;
  onError: () => void;
}

interface State {}

export class HLSPlayer extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);

    this.hls = new Hls();

    this.hls.on(Hls.Events.MANIFEST_PARSED, () => {
      if (this.videoEl) {
        this.videoEl.play();
      } else {
        console.error('manifest parsed before video el rendered');
      }
    });

    this.hls.on(Hls.Events.ERROR, (event, data) => {
      console.log(event, data);
      if (data.fatal) {
        this.props.onError();
      }
    });
  }

  private hls: Hls;
  private videoEl: HTMLVideoElement | null = null;

  public render() {
    return <video className="embeddedVideo" ref={this.setVideoRef} muted />;
  }

  public shouldComponentUpdate() {
    return false;
  }

  public componentDidMount() {
    this.handleLoadingVideo();
  }

  public componentDidUpdate() {
    this.handleLoadingVideo();
  }

  public componentWillUnmount() {
    if (this.videoEl) {
      this.videoEl.removeEventListener('error', this.onVideoError);
    }
    this.hls.destroy();
  }

  private handleLoadingVideo() {
    if (this.props.hlsURL !== '') {
      this.hls.loadSource(this.props.hlsURL);
    }
  }

  private setVideoRef = (video: HTMLVideoElement | null) => {
    if (video !== null) {
      this.videoEl = video;
      this.hls.attachMedia(video);

      this.videoEl.addEventListener('error', this.onVideoError);
    }
  };

  private onVideoError = () => {
    this.props.onError();
  };
}
