import * as React from 'react';

import { MosaicTile } from './tile';
import { FOCUS_REFRESH_RATE } from 'config';

import './style.css';

const ASPECT_RATIO = 16 / 9;

interface Props {
  children: Array<React.ReactNode>;
  height: number;
  width: number;
  onReplace: (idx: number) => void;
}

interface State {
  isFocused: boolean;
  focusSize: number;
  focusIdx: number;
}

export class Mosaic extends React.PureComponent<Props, State> {
  private cycleFocusTimer: number = -1;

  constructor(props: Props) {
    super(props);

    this.state = {
      isFocused: false,
      focusSize: 2,
      focusIdx: 0,
    };
  }

  public componentDidMount() {
    this.cycleFocusTimer = setInterval(() => {
      this.setState({
        isFocused: !this.state.isFocused,
        focusIdx: this.state.isFocused
          ? this.state.focusIdx
          : Math.floor(Math.random() * this.props.children.length),
      });
    }, FOCUS_REFRESH_RATE);
  }

  public componentWillUnmount() {
    clearInterval(this.cycleFocusTimer);
  }

  public render() {
    let aspectRatio = this.props.width / this.props.height;
    let isTooTall = aspectRatio < ASPECT_RATIO;

    // Try to make a square grid
    let tileCount = this.props.children.length;
    let columns = Math.ceil(Math.sqrt(tileCount));
    let rows = columns;

    // If its not square trim a col or row depending on closest aspect
    if (rows * columns > tileCount) {
      if (isTooTall) {
        rows -= 1;
      } else {
        columns -= 1;
      }
    }
    // Only render this many tiles
    let renderCount = rows * columns;

    // Calculate each tile size
    let tileWidth = this.props.width / columns;
    let tileHeight = tileWidth * (1 / ASPECT_RATIO);

    // If our trimming made us too short, then calculate off height
    if (tileHeight * rows < this.props.height) {
      tileHeight = this.props.height / rows;
      tileWidth = tileHeight * ASPECT_RATIO;
    }

    // Float us so we're centered.
    let xOffset = (tileWidth * columns - this.props.width) / 2;
    let yOffset = (tileHeight * rows - this.props.height) / 2;

    let styles = {
      top: -yOffset,
      left: -xOffset,
    };

    // Calculate Rows and Cols covered by Focus Video
    let focusRows = [];
    let focusCols = [];
    let focusRow = Math.floor(this.state.focusIdx / columns);
    let minRow = focusRow;
    if (focusRow + this.state.focusSize >= rows) {
      minRow = rows - this.state.focusSize;
    }

    let focusCol = this.state.focusIdx % columns;
    let minCol = focusCol;
    if (focusCol + this.state.focusSize >= columns) {
      minCol = columns - this.state.focusSize;
    }

    for (let i = 0; i < this.state.focusSize; i++) {
      focusRows.push(minRow + i);
      focusCols.push(minCol + i);
    }

    let tiles: Array<React.ReactNode> = [];
    for (let i = 0; i < renderCount; i++) {
      let row = Math.floor(i / columns);
      let col = i % columns;
      let xOffset = col * tileWidth;
      let yOffset = row * tileHeight;
      let width = tileWidth;
      let height = tileHeight;
      let obscured = false;
      let focused = false;

      if (this.state.isFocused) {
        if (this.state.focusIdx === i) {
          xOffset = Math.min(...focusCols) * width;
          yOffset = Math.min(...focusRows) * height;
          focused = true;
          width *= this.state.focusSize;
          height *= this.state.focusSize;
        } else {
          if (focusRows.indexOf(row) >= 0 && focusCols.indexOf(col) >= 0) {
            obscured = true;
          }
        }
      }

      tiles[i] = (
        <MosaicTile
          key={i}
          height={height}
          width={width}
          xOffset={xOffset}
          yOffset={yOffset}
          focused={focused}
          obscured={obscured}
          onReplace={() => {
            this.props.onReplace(i);
          }}
        >
          {this.props.children[i]}
        </MosaicTile>
      );
    }

    return (
      <div className="mosaic" style={styles}>
        {tiles}
      </div>
    );
  }
}
