import * as React from 'react';

import { CHANCE_TO_REPLACE } from 'config';

interface Props {
  children: React.ReactNode;
  height: number;
  width: number;
  xOffset: number;
  yOffset: number;
  focused: boolean;
  obscured: boolean;
  onReplace: () => void;
}

interface State {
  wasFocused: boolean;
}

// Sync with transition time in style.css
const ANIMATION_TIME = 1000;

export class MosaicTile extends React.PureComponent<Props, State> {
  private residualFloatTimer: number = -1;
  private replaceMeTimer: number = -1;

  constructor(props: Props) {
    super(props);

    this.state = {
      wasFocused: false,
    };
  }

  public componentWillReceiveProps(nextProps: Props) {
    if (nextProps.obscured && !this.props.obscured) {
      // We're becoming obscured... replace ourselves sometimes
      if (Math.random() < CHANCE_TO_REPLACE) {
        this.replaceMeTimer = setTimeout(() => {
          this.props.onReplace();
        }, ANIMATION_TIME);
      }
    }

    if (this.props.focused && !nextProps.focused) {
      this.setState(
        {
          wasFocused: true,
        },
        () => {
          // Schedule us to stop floating after animation
          this.residualFloatTimer = setTimeout(() => {
            this.setState({
              wasFocused: false,
            });
          }, ANIMATION_TIME);
        },
      );
    }
  }

  public componentWillUnmount() {
    clearTimeout(this.residualFloatTimer);
    clearTimeout(this.replaceMeTimer);
  }

  public render() {
    let zIndex = 0;
    if (this.props.focused) {
      zIndex = 2;
    } else if (this.state.wasFocused) {
      zIndex = 1;
    }

    let styles = {
      height: this.props.height,
      width: this.props.width,
      top: this.props.yOffset,
      left: this.props.xOffset,
      zIndex,
    };

    return (
      <div className="mosaic-tile" style={styles}>
        {this.props.children}
      </div>
    );
  }
}
