const path = require('path');

const webpack = require('webpack');
const HtmlWebpackPlugin = require('html-webpack-plugin');
const ForkTsCheckerWebpackPlugin = require('fork-ts-checker-webpack-plugin');

module.exports = function (env = {}) {
  let buildPath = path.join('dist', env.buildDir || env.build);
  let publicPath = env.publicPath || '/';

  let plugins = [
    new webpack.DefinePlugin({
      'process.env': {
        NODE_ENV: env.optimized && JSON.stringify('production'),
      },
    }),
    new HtmlWebpackPlugin({
      title: "Twitch Wall",
    }),
    new ForkTsCheckerWebpackPlugin({ checkSyntacticErrors: true }),
  ];

  let rules = [
    {
      test: /\.(ico|png|jpg|gif|svg|eot|ttf|woff|woff2|swf)$/,
      loader: 'file-loader',
      options: {
        name: '[name]-[hash].[ext]',
      }
    },
    {
      test: /\.tsx?$/,
      use: [
        { loader: 'cache-loader' },
        {
          loader: 'thread-loader',
          options: {
            workers: 2,
            poolTimeout: env.devserver ? Infinity : 500,
          },
        },
        {
          loader: 'ts-loader',
          options: {
            happyPackMode: true,
          }
        }
      ]
    },
    {
      test: /\.css$/,
      use: [
        'style-loader',
        {
          loader: 'css-loader',
          options: {
            minimize: env.optimized,
            sourceMap: true,
          }
        },
      ],
    },
  ];

  if (env.devserver && !env.optimized) {
    plugins.push(new webpack.HotModuleReplacementPlugin());
  }

  const stats = {
    assets: true,
    children: true,
    chunkModules: false,
    chunks: false,
    colors: true,
    hash: false,
    timings: false,
    version: false,
    modules: false,
  };

  let extraConfig = {};

  if (env.devserver) {
    extraConfig = {
      devServer: {
        compress: true,
        contentBase: './public/',
        historyApiFallback: {
          rewrites: [],
        },
        host: 'localhost',
        hot: !env.optimized,
        inline: !env.optimized,
        overlay: true,
        port: 8080,
        stats: env.optimized ? stats : 'errors-only',
      },
    };
  }

  return Object.assign({
    entry: {
      core: [path.join(__dirname, 'src/index.tsx')],
    },
    output: {
      chunkFilename: '[name]-[hash].js',
      crossOriginLoading: 'anonymous',
      filename: '[name]-[hash].js',
      hashDigestLength: 32,
      path: path.join(__dirname, buildPath),
      publicPath,
    },
    devtool: 'source-map',
    stats,
    resolve: {
      alias: {
        'wall': path.join(__dirname, 'src'),
        'config': path.join(__dirname, env.build === 'local'  ||  env.build == 'dev' ? 'config/local' : 'config/production'),
      },
      extensions: ['.js', '.jsx', '.ts', '.tsx'],
    },
    plugins,
    module: {
      rules,
    },
  }, extraConfig);
}
