import React, { Component } from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import { Row, Col, Button, Form, Select, Tag, Spin, Checkbox, notification } from 'antd';

import Image from './Image';
import { loadDataset, updateDefaultClass, updateShowExtra } from '../actions/dataset';
import YellerApi from '../apis/yeller_api';

import '../styles/Dataset.css';

class Dataset extends Component {
  static propTypes = {
    classes: PropTypes.array.isRequired,
    defaultClass: PropTypes.string,
    images: PropTypes.array.isRequired,
    loading: PropTypes.bool.isRequired,

    loadDataset: PropTypes.func.isRequired,
    updateDefaultClass: PropTypes.func.isRequired
  };

  loadDataset() {
    this.props.loadDataset(
      this.props.datasetSelector.database,
      this.props.datasetSelector.collection,
      this.props.images // need this otherwise you will be wasting time
    );
  }

  handleSubmitClick(e) {
    window.scrollTo(0, 0);

    const submission = this.props.images.map((data, i) => {
      return { ...data, label: data.label || this.props.defaultClass };
    });

    this.loadDataset();

    YellerApi.updateImageLabels(
      this.props.datasetSelector.database,
      this.props.datasetSelector.collection,
      submission
    );

    notification.success({ message: 'Dataset Submitted' });
  }

  renderImages() {
    if (this.props.loading) {
      return (
        <Row>
          <Spin size="large" />
        </Row>
      );
    }
    if (this.props.images && this.props.images.length) {
      return (
        <React.Fragment>
          <div className="dataset-image-container">
            {this.props.images.map((data, i) => {
              return <Image key={i} data={data} />;
            })}
          </div>
          <div className="dataset-submit-container">
            <Button type="primary" htmlType="submit" onClick={() => this.handleSubmitClick()}>
              Submit
            </Button>
          </div>
        </React.Fragment>
      );
    } else {
      return (
        <React.Fragment>
          <Row>
            <h1> Dataset is Empty ! </h1>
          </Row>
          <div className="dataset-submit-container">
            <Button type="primary" htmlType="reset" onClick={() => this.loadDataset()}>
              Refresh
            </Button>
          </div>
        </React.Fragment>
      );
    }
  }

  render() {
    return (
      <div className="dataset-container">
        <Row style={{ width: '80%', height: 80 }} type="flex" align="middle" justify="middle">
          <Col>
            <strong> Dataset: </strong>
            <Tag color="geekblue">{`${this.props.datasetSelector.database}:${
              this.props.datasetSelector.collection
              }`}</Tag>
          </Col>
          <Col>
            <strong> Matches: </strong>
            <Tag color="geekblue">{this.props.datasetSize}</Tag>
          </Col>
          <Col>
            <strong> Classes: </strong>
            {this.props.classes.map((klass, i) => {
              return (
                <Tag key={i} color="geekblue">
                  {klass}
                </Tag>
              );
            })}
          </Col>
          <Col>
            <Form layout="inline">
              <Form.Item label="Default Class">
                <Select
                  placeholder="Select"
                  style={{ width: 100 }}
                  value={this.props.defaultClass}
                  onChange={val => this.props.updateDefaultClass(val)}
                >
                  {this.props.classes.map((className, i) => {
                    return (
                      <Select.Option key={i} value={className}>
                        {className}
                      </Select.Option>
                    );
                  })}
                </Select>
              </Form.Item>
            </Form>
          </Col>
          <Col>
            <Form layout="inline">
              <Form.Item label="Extra">
                <Checkbox
                  checked={this.props.showExtra}
                  onChange={e => this.props.updateShowExtra(e.target.checked)}
                >
                  Show
                </Checkbox>
              </Form.Item>
            </Form>
          </Col>
        </Row>
        {this.renderImages()}
      </div>
    );
  }
}

const mapStateToProps = state => ({
  datasetSelector: state.datasetSelector,
  ...state.dataset
});

const mapDispatchToProps = { loadDataset, updateDefaultClass, updateShowExtra };

export default connect(
  mapStateToProps,
  mapDispatchToProps
)(Dataset);
