import React, { Component } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router-dom';
import PropTypes from 'prop-types';
import { Button, Icon, Form, AutoComplete, DatePicker, notification } from 'antd';
import { loadDataset } from '../actions/dataset';
import { updateDatasetSelection } from '../actions/datasetSelector';
import { DATABASES, COLLECTIONS } from '../constants';


class DatasetSelector extends Component {
  static propTypes = {
    history: PropTypes.object.isRequired,
    match: PropTypes.shape({
      params: PropTypes.shape({
        database: PropTypes.string,
        collection: PropTypes.string
      })
    })
  };

  componentDidMount() {
    if (this.props.match.params.database && this.props.match.params.collection) {
      this.props.updateDatasetSelection({
        database: this.props.match.params.database,
        collection: this.props.match.params.collection
      });
      this.props.loadDataset(
        this.props.match.params.database,
        this.props.match.params.collection
      );
    }
  }

  handleSubmit(e) {
    e.preventDefault();
    if (this.props.database && this.props.collection) {
      this.props.history.push(`/${this.props.database}/${this.props.collection}`);
      this.props.loadDataset(
        this.props.database,
        this.props.collection
      );
    } else {
      notification.error({
        message: 'Invalid Dataset',
        description: 'You must specify both database and collection'
      });
    }
  }

  render() {
    return (
      <Form layout="inline" onSubmit={e => this.handleSubmit(e)}>
        <Form.Item>
          <AutoComplete
            dataSource={DATABASES}
            defaultValue={this.props.match.params.database || this.props.database}
            placeholder="database"
            onChange={val => this.props.updateDatasetSelection({ database: val })}
          />
        </Form.Item>
        <Form.Item>
          <AutoComplete
            style={{width:400}}
            dataSource={COLLECTIONS}
            defaultValue={this.props.match.params.collection || this.props.collection}
            placeholder="collection"
            allowClear={true}
            filterOption={(inputValue, option) =>
              option.props.children.toUpperCase().indexOf(inputValue.toUpperCase()) !== -1
            }
            onChange={val => this.props.updateDatasetSelection({ collection: val })}
          />
        </Form.Item>
        <Form.Item>
          <Button type="primary" htmlType="submit">
            <Icon type="search" />
          </Button>
        </Form.Item>
      </Form>
    );
  }
}

const mapStateToProps = state => state.datasetSelector;
const mapDispatchToProps = { loadDataset, updateDatasetSelection };

export default withRouter(
  connect(
    mapStateToProps,
    mapDispatchToProps
  )(DatasetSelector)
);
