import json
import sys
import time
from threading import Thread

import boto3

from yeller.logger import log

def get_region_from_queue_url(queue_url):
    return queue_url.split(".")[1]

class SQSRunner(Thread):
    def __init__(self, queue_url, callback_func, sleep=3):
        Thread.__init__(self)
        region_name = get_region_from_queue_url(queue_url)
        sqs = boto3.resource('sqs', region_name=region_name)

        log.info('Initializing SQS Queue[{}]: {}'.format(region_name, queue_url))

        self.queue = sqs.Queue(queue_url)
        self.queue.load()
        self.callback_func = callback_func
        self.sleep_timeout = sleep
        self.run_loop = True

    def loop_func(self):
        log.debug("SQS.get_messages from {}".format(self.queue.url))
        items = []
        messages = self.queue.receive_messages(MaxNumberOfMessages=10,
                                               VisibilityTimeout=10,
                                               WaitTimeSeconds=0)
        for message in messages:
            items.append(json.loads(message.body))
            message.delete()

        log.debug("{} got {} messages from sqs".format(self.queue.url, len(items)))
        return items

    def run(self):
        while self.run_loop:
            try:
                log.debug("{} run_loop".format(self.queue.url))
                items = self.loop_func()
                if items:
                    self.callback_func(items)
            except KeyboardInterrupt as e:
                self.run_loop = False
                raise e
            except Exception:
                log.exception("failed to run SQS SQSRunner Loop")

            time.sleep(1)

    def stop(self):
        self.run_loop = False


class SQS(object):
    def __init__(self, queue_urls):
        log.info("Initializing SQS Queues: {}".format(queue_urls))
        self.runners = []
        for queue_url in queue_urls:
            runner = SQSRunner(queue_url, self.process_messages)
            runner.daemon = True
            self.runners.append(runner)

    def process_messages(self, messages):
        raise NotImplementedError

    def start_loop(self):
        log.info("SQS.start_loop")
        for runner in self.runners:
            runner.start()

    def stop_loop(self):
        log.info("SQS.start_loop")
        for runner in self.runners:
            runner.stop()
