import pprint
import tempfile
import urllib.request
import uuid
import os
import json

import boto3
import falcon
import requests

from yeller.config import config
from yeller.logger import log


def download_url(url):
    _handle, filename = tempfile.mkstemp(".mp4")
    urllib.request.urlretrieve(
        url, filename)
    return filename


def download_twitch_clip(twitch_clip):
    slug = twitch_clip.split('/')[-1]
    headers = {"Client-ID": config.TWITCH_CLIENT_ID}
    result = requests.get('https://api.twitch.tv/helix/clips',
                          params={'id': slug}, headers=headers).json()
    log.info(pprint.pformat(result))
    log.info("Twitch Clip: {}".format(result['data'][0]['title']))
    url = result['data'][0]['thumbnail_url'].split(
        "-preview-")[0] + '.mp4'
    log.info("Twitch Url: {}".format(url))
    return download_url(url)


def download_youtube(youtube_url):
    _handle, filename = tempfile.mkstemp(".mp4")
    os.system('youtube-dl --no-continue {} -o {}'.format(youtube_url, filename))
    return filename


def queue_file(detector, filename):
    extension = os.path.splitext(filename)[1]

    sqs = boto3.resource('sqs', region_name=config.SQS_REGION)
    queue = sqs.Queue(config.SQS_UPLOAD[0])

    s3 = boto3.client("s3")
    upload_key = "upload/data/{}.{}".format(str(uuid.uuid4()), extension)
    s3.upload_file(filename, config.GAMESENSE_BUCKET, upload_key)

    queue.send_message(MessageBody=json.dumps({
        'detector': detector,
        's3_bucket': config.GAMESENSE_BUCKET,
        's3_key': upload_key
    }))

    os.remove(filename)
    log.info("File queued: {}".format(filename))


class UploadFile:
    def on_post(self, req, resp):
        f = req.get_param('file')
        detector = req.get_param('detector')

        log.info("{}: {}".format(detector, f.filename))

        extension = os.path.splitext(f.filename)[1]
        _, temp = tempfile.mkstemp(extension)
        with open(temp, 'wb') as tf:
            tf.write(f.file.read())

        queue_file(detector, temp)

        resp.media = {'ok': True}


class UploadURL:
    def on_post(self, req, resp):
        detector = req.media.get('detector')
        url = req.media.get('twitchClipUrl')

        log.info("{}: {}".format(detector, url))

        if url.startswith('https://clips.twitch.tv/'):
            filename = download_twitch_clip(url)
        elif url.startswith('https://www.youtube.com'):
            filename = download_youtube(url)
        elif url.startswith('http'):
            filename = download_url(url)
        else:
            resp.status = falcon.HTTP_400
            return

        queue_file(detector, filename)

        resp.media = {'ok': True}
