import os
import tempfile
import uuid
import shutil

import boto3
import cv2

from yeller.classes.mongo import MongoHeler
from yeller.classes.sqs import SQS
from yeller.config import config
from yeller.logger import log

mongo_helper = MongoHeler()


class UploadWorker(SQS):
    def __init__(self):
        super().__init__(config.SQS_UPLOAD)

    def process_messages(self, messages):
        for m in messages:
            self.process_file(m['detector'], m['s3_bucket'], m['s3_key'])

    def upload_image(self, detector, filename):
        s3 = boto3.client("s3")
        upload_key = "upload/{}/{}.jpg".format(detector, str(uuid.uuid4()))
        s3.upload_file(filename, config.GAMESENSE_BUCKET, upload_key)

        doc = {"bucket_name": config.GAMESENSE_BUCKET, "key": upload_key}
        mongo_helper.prepare_for_write(doc)
        mongo_helper.get_collection('upload', detector).insert_one(doc)

    def process_file(self, detector, bucket, key):
        extension = os.path.splitext(key)[1]
        _, filename = tempfile.mkstemp(extension)

        s3 = boto3.client("s3")
        s3.download_file(bucket, key, filename)

        log.info("Processing File: {}".format(filename))

        if extension == '.mp4':
            video = cv2.VideoCapture(filename)
            save_fps = 4
            n_skip = int(video.get(cv2.CAP_PROP_FPS) / save_fps)

            tempdir = tempfile.mkdtemp()
            files = []
            idx = 0
            while True:
                success, image = video.read()
                if not success:
                    break

                fn = os.path.join(tempdir, '{}.jpg'.format(idx))
                idx += 1

                files.append(fn)
                cv2.imwrite(fn, image)

                for _ in range(n_skip):
                    success, image = video.read()
                    if not success:
                        break

            for fn in files:
                self.upload_image(detector, fn)

            shutil.rmtree(tempdir)
        else:
            self.upload_image(detector, filename)

        os.remove(filename)
        log.info("Processing Done: {}".format(filename))
