module Authorization
  extend ActiveSupport::Concern

  class NotAuthenticated < Exception; end
  class Forbidden < Exception; end

  # User _making_ the request.
  def user_id
    @user_id ||= begin
      uid = request.headers['User-ID']
      if uid && !uid.to_s.match(/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i)
        raise ActionController::ParameterMissing.new('user_id is not a valid UUID')
      end
      uid
    end
  end

  # Person object for user _making_ the request.
  def current_person
    @current_person ||= begin
      p = Person.where(user_id: user_id).first_or_create!
      p.roles = request.headers['X-Roles']&.split(',') || []
      p
    end
  end

  def require_abilities!(*abilities)
    _require_user!
    return if current_person.is_admin?
    options = abilities.extract_options!
    obj = options.delete(:for)
    abilities.each do |ability|
      unless current_person.send("can_#{ability}?", obj)
        raise Forbidden.new("User is not allowed to #{ability.to_s.gsub('_',' ')}")
      end
    end
  end

  module ClassMethods
    # These class methods are called _outside_ the scope of an action, so render errors
    # (and thereby prevent action from starting) instead of raising an exception that
    # would bypass `BaseController.handle_exceptions`.

    def require_user!(opts)
      prepend_before_action(opts) do
        render_on_error { _require_user! }
      end
    end

    def require_admin!(opts)
      before_action(opts) do
        render_on_error do
          _require_user!
          unless current_person.is_admin?
            render status: 403,
              json: {status: 403, error: 'Admin privileges required'}
          end
        end
      end
    end

  end

  private

  # For the sake of sharing logic with other methods that need it executed in place
  # (as opposed to in a `before_action`).
  def _require_user!
    raise NotAuthenticated.new('Missing User-ID header') if user_id.blank?
  end

  def render_on_error
    yield
  rescue ActionController::ParameterMissing => e
    render status: 400, json: {status: 400, error: e.message}
  rescue NotAuthenticated => e
    render status: 401, json: {status: 401, error: e.message}
  end
end
