class V1::ArticlesController < V1::BaseController
  include Authorization

  require_user! except: [:index, :show]

  def index
    filters    = filter_params
    pub_filter = filters.delete(:published_before)
    order      = filters.delete(:order)

    articles = Article
    articles = articles.where('published_at < ?', pub_filter) if pub_filter
    articles = articles.order(order).filter(filters)

    render json: articles
  end

  def show
    article = Article.friendly.find(params[:id])
    unless article.is_published?
      raise Forbidden if user_id.blank? ||
        (!current_person.is_admin? && current_person != article.author)
    end
    render json: article
  end

  def create
    # Allow an article to be created in the 'review' state for convenience,
    # e.g. for some flow like "New Article" -> edit -> "Submit For Review" button.
    eparams = entry_params
    if ![nil, 'draft', 'review'].include?(eparams[:state])
      raise ActionController::BadRequest.new('new articles may only have states: draft, review')
    end
    eparams[:aasm_state] = eparams.delete(:state)
    article = Article.create!(eparams.merge(author_id: current_person.id))
    render json: article
  end

  def update
    if params[:state] == 'final' && !current_person.is_admin?
      raise Forbidden.new('only admins can approve articles')
    end
    article = Article.find(params[:id])
    # Ignore title and/or body updates if they haven't actually changed.
    params.delete(:title) if params[:title] == article.title
    params.delete(:body) if params[:body] == article.body
    article.do_update!(entry_params)
    render json: article
  rescue AASM::InvalidTransition
    content_changes = !params.slice(:title, :body).empty?
    target_state = params[:state] || article.state
    msg = "cannot transition #{article.state} -> #{target_state}#{' with content changes' if content_changes}"
    raise ActionController::BadRequest.new(msg)
  end

  def destroy
    article = Article.find(params[:id])
    render json: article.destroy! and return if current_person.is_admin?

    if article.is_published?
      raise Forbidden.new('only admins can delete published articles')
    elsif article.author != current_person
      raise Forbidden
    end
    render json: article.destroy!
  end

  private

  def filter_params
    fparams = params.permit(:limit, :offset, :parent_id, :parent_type, :ids, :flags, ids: [], flags: [])
    fparams[:flags] = fparams[:flags].split(',') if fparams[:flags].is_a?(String)

    if fparams.slice(:parent_id, :parent_type).keys.length == 1
      raise ActionController::BadRequest.new('parent filter must specify both id and type')
    elsif ptype = fparams[:parent_type]
      begin
        ptype.camelize.constantize
      rescue NameError
        raise ActionController::BadRequest.new("invalid parent type: #{ptype.inspect}")
      end
      fparams[:parent_type] = ptype.camelize
    end

    user_is_author = !user_id.blank? && (params[:author_id] == user_id || current_person.is_admin?)
    fparams[:published_before] = Time.now unless user_is_author
    fparams[:order] = user_is_author ? 'updated_at DESC' : 'published_at DESC'

    if params.include?(:state)
      if params[:state] != 'final' && !user_is_author
        raise Forbidden.new('unpublished articles may only be viewed by their authors')
      end
      fparams[:aasm_state] = params[:state]
    elsif !user_is_author
      # Default view for authors viewing their own articles is _all_ articles;
      # default view for everyone else is published articles only.
      fparams[:aasm_state] = 'final'
    end

    if params.include?(:author_id)
      # API clients will always refer to people by their *User* ID, not their *Person* ID.
      fparams[:author] = Person.find_by!(user_id: params[:author_id])
    end

    fparams
  end

  def entry_params
    eparams = params.permit(:title, :body, :cover, :remote_cover_url, :state, :slug,
                            :parent_id, :parent_type, :published_at, :flags, flags: [])
    eparams[:flags] = eparams[:flags].split(',') if eparams[:flags].is_a?(String)

    if eparams[:cover] && ![ActionController::Parameters,
                            ActionDispatch::Http::UploadedFile].include?(eparams[:cover].class)
      raise ActionController::BadRequest.new('Article cover must be an image')
    end

    if eparams.values_at(:parent_id, :parent_type).compact.length == 1
      raise ActionController::BadRequest.new('Article parent must have both parent_id and parent_type')
    end

    eparams
  end
end
