class V1::BaseController < ApplicationController
  around_action :handle_exceptions
  before_action :serializer_context

  attr_accessor :errors

  def curse
    @curse ||= Curse.new
  end

  private

  def serializer_context
    V1::BasePresenter.controller = self
  end

  def handle_exceptions
    begin
      yield
    rescue ActionController::BadRequest,
        ActionController::ParameterMissing,
        ActiveRecord::RecordNotDestroyed,
        ActiveRecord::RecordInvalid => e
      context = {}
      if e.respond_to?(:record) && e.record
        context.merge!({ errors: e.record.errors.to_hash(true) })
      end
      render error(400, e.message, context)
    rescue ActiveRecord::RecordNotUnique => e
      # `PG::` exceptions are nothing more than `StandardError`s with Postgres' error
      # reply crammed into the message, but we can at least try to extract something helpful.
      if e.cause&.message =~ /Key \(([^)]+)\)=.* already exists/
        render error(400, "Record already exists with this #{$1}")
      else
        render error(400, 'Record already exists')
      end
    rescue MiniMagick::Invalid, MiniMagick::Error => e
      # MiniMagick is yet another Ruby library that has no helpful repackaging of errors
      # apart from cramming the full output buffer into `message`.
      extra = {}
      extra['minimagick'] = e.message unless Rails.env.production?
      render error(400, 'Invalid image transformation', extra)
    rescue Authorization::NotAuthenticated => e
      render error(401, e.message)
    rescue Authorization::Forbidden => e
      render error(403, e.message)
    rescue ActiveRecord::RecordNotFound => e
      render error(404, e.message)
    rescue Curse::CurseAPIError => e
      render error(502, e.message)
    end
  end

  def wrap_in_transaction
    ActiveRecord::Base.transaction { yield }
  end

  # Canned Responses

  def ok
    { json: {}, status: 200 }
  end

  def error(status, msg, extra={})
    { status: status, json: {status: status, error: msg}.merge(extra) }
  end
end
