class V1::ImagesController < V1::BaseController
  include Authorization

  require_admin! only: [:destroy]

  def index
    render json: Image.filter(index_params)
  end

  def show
    # Rails helpfully chops off the extension in the URL,
    # e.g. "/v1/images/foo.bar" -> params[:id] == "foo" && params[:format] == "bar"
    if params.include?(:format)
      data = image.transform(transform_params.merge(type: params[:format]))
      # Some MiniMagick errors raise an exception (caught by BaseController),
      # but some just make the output empty
      raise ActionController::BadRequest.new('Invalid image transformation') if data.blank?
      send_data data,
        type: FileMagic.new(:mime_type).buffer(data),
        disposition: 'inline'
    else
      render json: image
    end
  rescue NoMethodError => e
    # TODO: This obviously needs to be less janky as fuck.
    # This is the nonsense CarrierWave throws when an Image record refers to an S3 asset
    # that no longer exists:
    if e.message == "undefined method `body' for nil:NilClass" \
        && e.backtrace[0] =~ %r{carrierwave/storage/fog\.rb:277:in `read'}
      image.destroy
      raise ActiveRecord::RecordNotFound.new('image asset not found')
    else
      raise e
    end
  end

  def create
    cparams = create_params
    case request.media_type
    when 'multipart/form-data'
      unless params.require(:image).class == ActionDispatch::Http::UploadedFile
        raise ActionController::BadRequest.new('Image must be an uploaded file')
      end
      # No changes needed to `cparams`.
    when /^image\//
      # CarrierWave recommends this stupid hack on their GitHub Wiki:
      # (we never use original filename; we just set filename to the file's SHA1 sum)
      file = request.body
      def file.original_filename; "doesn't matter"; end
      cparams[:image] = file
    else
      # HTTP 415: Unsupported Media Type
      render error(415, 'expected content type "image/*" or "multipart/form-data"')
      return
    end

    new_image = Image.new(cparams)
    if existing_image = Image.find_by(id: new_image.sha1sum)
      render json: existing_image
    else
      new_image.save!
      render json: new_image
    end
  end

  def update
    image.update!(update_params)
    render json: image
  end

  def destroy
    render json: image.destroy!
  end

  private

  def image
    @image ||= Image.find(params.require(:id))
  end

  def index_params
    params.permit(:tag, :limit, :offset, tags: [], ids: [])
  end

  def transform_params
    transforms = params.except(:controller, :action, :id, :image, :format)
    allowed = transforms.permit(:size)
    unless (invalid = transforms.except(*allowed.keys)).empty?
      raise ActionController::BadRequest.new("unsupported transforms: #{invalid.keys.join(', ')}")
    end
    allowed
  end

  def create_params
    params.permit(:image, :tags, tags: [], metadata: [])
  end

  def update_params
    params.permit(tags: [], metadata: [])
  end
end
