class V1::MatchesController < V1::BaseController
  include Authorization

  require_user! only: [:update]
  require_admin! only: [:create, :destroy]

  def index
    matches = params[:user_id] ? Match.joins(:opponents) : Match
    matches = matches.filter(filter_params)
    render json: matches
  end

  def show
    render json: match
  end

  def update
    if current_person.is_admin?
      update_params = match_params
      update_params[:state] = 'locked' if update_params[:winner_id]
      match.update!(update_params)
      render json: match and return
    end

    # The only thing non-admins can try to change is :winner_id.
    winner_id = match_params[:winner_id]
    raise Forbidden if winner_id.nil? or !match_params.to_h.except(:id, :winner_id).empty?

    # Dumb edge case (which may even happen more in testing than prod):
    # possible for the user to be an owner and/or captain of more than one participating team.
    opponents = match.opponents.select do |opponent|
      current_person.can_modify_team?(opponent.contender)
    end
    raise Forbidden if opponents.empty?
    opponents.each do |opponent|
      opponent.update!(winner_id: winner_id)
    end
    render json: match.reload
  end

  def create
    match = Match.create!(match_params)
    render json: match
  end

  def destroy
    match.destroy!
    render json: match
  end

  private

  def match
    @match ||= Match.find(params.delete(:id))
  end

  def match_params
    params[:opponents_attributes] = params.to_unsafe_h[:opponents]
    params.permit(Match.permitted_attributes)
  end

  def filter_params
    filters = params.permit(:game_id, :series_id, :winner_id,
                            :limit, :offset, :ids, ids: [])
    if params.include?(:user_id)
      person = Person.find_by!(user_id: params[:user_id])
      filters['opponents.contender_id'] = person.team_ids + [person.id]
    end
    filters
  end
end
