class V1::SeriesController < V1::BaseController
  include Authorization

  require_admin! only: [:update, :create, :destroy]
  around_action :wrap_in_transaction, only: :create

  def index
    series = Series

    if params[:user_id]
      series = series.joins(
        'LEFT OUTER JOIN matches ON series.id = matches.series_id ' +
        'JOIN opponents ON (series.id = opponents.contest_id OR matches.id = opponents.contest_id)'
      ).distinct
    end

    filters = filter_params
    series = series.where('scheduled_at >= ?', filters.delete(:from)) if filters[:from]
    series = series.where('scheduled_at <= ?', filters.delete(:to)) if filters[:to]
    series = series.filter(filter_params).order('scheduled_at ASC').distinct
    render json: series
  end

  def show
    series = Series.find(params[:id])
    render json: series
  end

  def update
    series = Series.find(params[:id])
    series.update!(update_params)
    # TODO look at how to trigger the entire validation chain just via #update!
    series.save!
    render json: series
  end

  def create
    series = Series.create!(create_params)
    unless params[:skip_curse]
      opponents = series.opponents.map(&:contender).map(&:name).uniq.sort
      name = "#{series.game.abbr+': ' if series.game}#{opponents.join(' vs ')}"
      series.settings['twitch_voice_url'] = curse.channel(create: true, name: name).invite
      series.save!
    end
    render json: series
  end

  def destroy
    series = Series.find(params[:id])
    series.destroy!
    render json: series
  end

  private

  def create_params
    return @create_params if @create_params
    @create_params = update_params.merge(
      tournament_stage_id: params[:tournament_stage_id],
      game_id: TournamentStage.find(params.require(:tournament_stage_id)).tournament.game_id,
    )
  end

  def update_params
    return @update_params if @update_params

    params[:opponents_attributes] = params.to_unsafe_h[:opponents]
    if params[:matches]
      params[:matches_attributes] = params.to_unsafe_h[:matches].collect do |match|
        match[:opponents_attributes] = match[:opponents]
        match
      end
    end
    @update_params = params.permit(
      :type, :scheduled_at,
      matches_attributes: Match.permitted_attributes,
      opponents_attributes: Opponent.permitted_attributes
    )
  end

  def filter_params
    return @filter_params if @filter_params

    fparams = params.permit(:game_id, :tournament_stage_id, :on, :from, :to, :limit, :offset, :ids, ids: [])

    # Filters for Season and Tournament need to consult parent models,
    # because Series has neither `season_id` nor `tournament_id`.
    # However, it's pointless to filter on Tournaments if there's already a filter on Stages,
    # and likewise to filter on Series if there's already a filter for Tournaments or Stages.
    if !fparams[:tournament_stage_id]
      stage_ids = case
        when params[:tournament_id]
          [Tournament.find(params[:tournament_id])]
        when params[:tournament_ids].is_a?(Array)
          Tournament.find(params[:tournament_ids])
        when params[:tournament_ids].is_a?(String)
          Tournament.find(params[:tournament_ids].split(','))
        when params[:season_id]
          Season.find(params[:season_id]).tournaments
        when params[:league_id]
          League.find(params[:league_id]).seasons.map(&:tournaments).flatten
        when params[:league_name]
          League.find_by(name: params[:league_name]).seasons.map(&:tournaments).flatten
        else []
        end.map(&:stage_ids).flatten
      fparams[:tournament_stage_id] = stage_ids unless stage_ids.empty?
    end

    if params[:user_id]
      person = Person.find_by!(user_id: params[:user_id])
      fparams['opponents.contender_id'] = person.team_ids + [person.id]
    end

    if fparams[:on]
      if fparams[:from] || fparams[:to]
        raise ActionController::BadRequest.new('cannot combine "on" with "from" or "to"')
      end
      time = parse_time(fparams.delete(:on))
      fparams[:from] = time.beginning_of_day
      fparams[:to] = time.end_of_day
    end

    if fparams[:from]
      fparams[:from] = parse_time(fparams[:from]).beginning_of_day
    end
    if fparams[:to]
      fparams[:to] = parse_time(fparams[:to]).end_of_day
    end

    @filter_params = fparams
  end

  def parse_time(input)
    return input if input.is_a?(Time)
    Time.parse(input)
  rescue ArgumentError => e
    raise ActionController::BadRequest.new("invalid time/date: #{input.inspect}")
  end
end
