class V1::TeamsController < V1::BaseController
  include Authorization

  require_user! only: [:create, :update]
  require_admin! only: [:destroy]

  def index
    teams = (params[:user_id] ? person.teams
              : params[:hidden] ? Team
              : Team.where("settings->'hidden' IS NULL OR settings->'hidden' != 'true'")).
            filter(filter_params).
            includes(:people, tournament_entries: :tournament)
    render json: teams
  end

  def show
    render json: team
  end

  def create
    unless current_person.is_admin?
      case
      when current_person.organization_ids.empty?
        raise Forbidden.new("User must have verified email to create a team")
      when params[:organization_id].blank?
        # When not given an organization, default to the owner's.
        # TODO: Choose a sensible default when people can _actually_ be part of many orgs.
        params[:organization_id] = current_person.organization_ids.first
      when !current_person.is_admin? && !current_person.organization_ids.include?(params[:organization_id])
        raise Forbidden.new('Cannot make a team for an organization you are not a member of')
      end
    end

    team = Team.create!(team_params)
    team.add_member!(current_person, ['owner'])
    render json: team
  end

  def update
    require_abilities! :modify_team, for: team
    team.update!(team_params)
    render json: team
  end

  def destroy
    team.destroy!
    render json: team
  end

  private

  def team
    @team ||= Team.friendly.find(params.require(:team_id))
  end

  def person
    @person ||= Person.find_by!(user_id: params[:user_id])
  end

  def team_params
    return @team_params if @team_params

    [:cover, :logo].each do |param|
      if params[param] && ![ActionController::Parameters,
                            ActionDispatch::Http::UploadedFile].include?(params[param].class)
        raise ActionController::BadRequest.new("#{param.to_s.capitalize} must be an image")
      end
    end

    @team_params = params.permit(:name, :slug, :description, :website, :country, :organization_id,
                                 :logo, :remote_logo_url, :cover, :remote_cover_url,
                                 settings: [:charity, :hidden], social: Sociable::NETWORKS)
  end

  def filter_params
    params.permit(:organization_id, :limit, :offset, :ids, ids: [])
  end
end
