class Article < ApplicationRecord
  include AASM
  include FriendlyId

  belongs_to :author, class_name: 'Person', foreign_key: 'author_id'
  belongs_to :parent, polymorphic: true, optional: true

  mount_uploader :cover, ArticleCoverUploader
  validates_integrity_of :cover
  validates_processing_of :cover
  validates_download_of :cover

  validates :title, presence: true
  validates :body, presence: true

  before_validation do |article|
    if article.parent_id_changed? || article.parent_type_changed?
      # Bindings to make this whole block easier to read:
      id_changed = article.parent_id_changed?
      type_changed = article.parent_type_changed?
      new_id = id_changed ? article.changes[:parent_id][1] : article.parent_id
      new_type = type_changed ? article.changes[:parent_type][1]&.camelize : article.parent_type

      # Correct the capitalization of the new parent type while we're at it:
      article.parent_type = new_type if type_changed && !new_type.blank?

      if (id_changed && new_id.blank? && !type_changed) ||
         (type_changed && new_type.blank? && !id_changed)
        # If one was nil'd and the other wasn't changed at all,
        # assume the user meant to remove the association entirely.
        article.parent_id = article.parent_type = article.parent = nil
      elsif (id_changed && !new_id.blank?) || (type_changed && !new_type.blank?)
        # If either was updated to non-nil, ensure that _both_ are non-nil.
        if new_id.blank? || new_type.blank?
          errors.add(:parent, 'update requires both parent_type and parent_id')
        else
          # Non-nil parent; make sure it's a valid reference.
          begin
            model = new_type.camelize.constantize
            errors.add(:parent, 'does not exist') unless model.exists?(new_id)
          rescue NameError
            errors.add(:parent, "has invalid type: #{new_type.inspect}")
          end
        end
      end
    end

    throw(:abort) unless errors.empty?
  end

  before_save do |article|
    article.flags.uniq! if article.flags_changed?
  end

  friendly_id :title, use: :slugged

  def should_generate_new_friendly_id?
    aasm_state_changed? && state == 'final' && slug.blank?
  end

  aasm whiny_transactions: true do
    state :draft, initial: true
    state :review, :final

    # AASM's callbacks are a little inconvenient; here's how this works:
    #
    # 1. This event's `before` block is called, caching args to a temporary instance variable.
    #
    # 2. AASM steps through transitions to see if any are possible.
    #    In doing so, it calls these `if` guards (defined below as private methods), which
    #    consult `@aasm_changes`.
    #
    # 3. After AASM finds the right transition, but before it persists the new state to the DB,
    #    the (unfortunately global) `after_all_transitions` block is called, which in turn
    #    triggers ActiveRecord validations.
    #    If AR validations fail, the whole operation is aborted and an AR exception is raised.
    #
    # 4. If `after_all_transitions` finishes successfully, the new AASM state is also persisted.
    #
    # 5. Lastly, whether everything worked or not, `@aasm_changes` is nil'd out.
    #
    # Two minor annoyances:
    #   - updating the record and persisting the new state are each their own DB operation
    #     (hard to combine them into one without just hacking around AASM)
    #   - the return value for a success is just `true`
    #     (but at least the instance doesn't need to be reloaded afterward)
    after_all_transitions ->{ update!(@aasm_changes.except(:state)) }
    event :do_update,
      before: lambda {|changes={}| @aasm_changes = changes},
      ensure: ->{ @aasm_changes = nil } \
    do
      transitions from: :draft, to: :draft, if: :draft_to_draft?
      transitions from: :draft, to: :review, if: :draft_to_review?
      transitions from: :draft, to: :final, if: :draft_to_final?

      transitions from: :review, to: :draft, if: :review_to_draft?
      transitions from: :review, to: :review, if: :review_to_review?
      transitions from: :review, to: :final, if: :review_to_final?

      transitions from: :final, to: :final, if: :final_to_final?
    end
  end

  def self.states
    self.aasm.states.map{|s|s.name.to_s}
  end

  def state
    read_attribute(:aasm_state)
  end

  def is_published?
    state == 'final' && !published_at.nil? && published_at <= Time.now
  end

  private

  def content_changed?
    !@aasm_changes.slice(:title, :body).empty?
  end

  def new_state_nil_or?(state)
    [nil, state].include?(@aasm_changes[:state])
  end

  def draft_to_draft?
    new_state_nil_or?('draft')
  end

  def draft_to_review?
    @aasm_changes[:state] == 'review'
  end

  def draft_to_final?
    @aasm_changes[:state] == 'final'
  end

  def review_to_draft?
    @aasm_changes[:state] == 'draft' \
      || (@aasm_changes[:state].nil? && content_changed?)
  end

  def review_to_review?
    new_state_nil_or?('review')
  end

  def review_to_final?
    @aasm_changes[:state] == 'final' && !content_changed?
  end

  def final_to_final?
    new_state_nil_or?('final')
  end
end
