class Image < ApplicationRecord
  mount_uploader :image, ImageUploader

  before_validation(on: :create) do
    self.id ||= self.sha1sum
  end

  def data
    # Lazily loads contents into memory, in case we want to do multiple operations
    # that'd otherwise cause a disk read for each one.
    return @data if defined?(@data)
    image.cache_stored_file! unless image.cached?
    @data = self.image.read
  rescue Errno::ENOENT
    @data = ""
  end

  def sha1sum
    @sha1sum ||= Digest::SHA1.hexdigest(data)
  end

  def content_type
    return nil if data.blank?
    @content_type ||= FileMagic.new(:mime_type).buffer(data)
  end

  def format
    return nil if content_type.blank?
    @format ||= content_type.split('/')[1]
  end

  def url
    "#{image.url}"
  end

  TRANSFORMS = [:size, :type]

  def transform(changes)
    if (unsupported = changes.keys.map(&:to_sym) - TRANSFORMS).length > 0
      raise "unsupported transforms: #{unsupported.map(&:inspect).join(', ')}"
    end

    changes[:type] = 'jpeg' if changes[:type] == 'jpg'
    if changes.empty? || (changes.except(:type).empty? && changes[:type] == format)
      return data
    end

    # Need an output type, so use the current one if it wasn't specified.
    changes[:type] = format unless changes.include?(:type)

    # Retrieve it from remote store if not yet done so.
    image.cache_stored_file! unless image.cached?

    # We should do some fancy dispatching when we support more transformations.
    MiniMagick::Tool::Convert.new do |convert|
      convert << image.path
      convert.resize changes[:size] if changes[:size]
      convert << "#{changes[:type]}:-"
      # convert.call(stdin: data)
    end
  end
end
