class Match < ApplicationRecord
  attribute :scheduled_at, :datetime

  belongs_to :series, required: false
  belongs_to :game, required: false
  belongs_to :winner, class_name: 'Opponent', required: false
  attribute :state, :string
  has_many :opponents, as: :contest, inverse_of: :contest
  has_many :teams, through: :opponents, source: :contender, source_type: 'Team'
  has_many :people, through: :opponents, source: :contender, source_type: 'Person'

  accepts_nested_attributes_for :opponents, allow_destroy: true
  validate :validate_opponents

  before_destroy :validate_series

  def self.permitted_attributes
    [:id, :_destroy, :game_id, :series_id, :winner_id,
     :scheduled_at, opponents_attributes: Opponent.permitted_attributes]
  end

  # Opponent model calls its Match's `determine_winner!` when its winner is updated.
  def determine_winner!
    # Only an admin action would change Match state after it's locked or contested,
    # and the controller already just does `match.update!(state:'locked')` in that case.
    return if ['locked','contested'].include?(state)
    winners = opponents.collect(&:winner_id)

    case
    when winner_id.nil?
      # this was the first vote
      update!(winner_id: winners.compact[0])
    when winners.compact.any? {|w| w != winner_id}
      # someone disagrees / changed their vote
      update!(state: 'contested')
    when state == '' && !winners.include?(nil)
      # all votes are in and none disagree (from above)
      update!(state: 'decided')
    end
  end

  private

  def validate_opponents
    errors.add(:opponents, 'requires at least two') if opponents.size < 2
  end

  def validate_series
    if series && series.invalid?
      @errors = series.errors
      throw(:abort)
    end
  end
end
