class Opponent < ApplicationRecord
  # "contest" is either a match or a series
  belongs_to :contest, polymorphic: true, inverse_of: :opponents
  # "contender" is either a team, or a person participating solo
  belongs_to :contender, polymorphic: true, inverse_of: :opponents
  # "winner" is the Opponent that self thinks won
  belongs_to :winner, class_name: 'Opponent', required: false

  before_commit :validate_contest
  before_destroy :validate_contest

  def self.permitted_attributes
    [:id, :_destroy, :winner_id,
     :contest_id, :contest_type,
     :contender_id, :contender_type]
  end

  # Make sure `winner_id` refers to an Opponent actually in the match.
  validate do |opponent|
    unless opponent.winner_id.blank? or contest.opponent_ids.include?(opponent.winner_id)
      errors.add(:winner, "did not participate in this match")
    end
  end

  # Update parent Match when an Opponent votes on a winner.
  after_save do |opponent|
    opponent.contest.determine_winner! if opponent.winner_id_changed?
  end

  private

  # Make sure we raise relevant association errors on contest#update and abort
  # the transaction.
  def validate_contest
    if contest && contest.invalid?
      @errors = contest.errors
      throw(:abort)
    end
  end
end
