class Person < ApplicationRecord
  include Sociable

  has_and_belongs_to_many :organizations
  has_many :team_memberships, class_name: 'TeamMember', dependent: :destroy
  has_many :teams, through: :team_memberships
  has_many :tournament_entries, as: :entrant
  has_many :tournaments, through: :tournament_entries
  has_many :opponents, as: :contender, inverse_of: :contender
  has_many :articles, inverse_of: 'author'

  validates :user_id, uniqueness: true,
            format: { with: /\A[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}\z/,
                      message: 'user_id must be a valid v4 UUID' }
  validate_social :social

  mount_uploader :photo, PersonPhotoUploader
  validates_integrity_of :photo
  validates_processing_of :photo
  validates_download_of :photo

  def self.find_by_id(id)
    Person.where('id = ? OR user_id = ?', id, id).first!
  end

  # While edb tracks a team _member_'s roles on a per-team basis,
  # a _person_'s roles are only ever set by controllers based on HTTP headers.
  attr_writer :roles
  def roles
    @roles ||= []
  end

  def is_admin?
    roles.include?('admin')
  end

  def can_modify_team?(team)
    is_admin? || TeamMember.can_modify?(self, team)
  end

  def can_modify_tournament_entry?(entry)
    return true if is_admin?
    entry = entry.is_a?(TournamentEntry) ? entry : TournamentEntry.find_by!(id: entry)
    case entry.entrant_type
      when 'Team' then can_modify_team?(entry.entrant_id)
      else false
    end
  rescue
    false
  end

  def can_modify_image?(image)
    return true if is_admin?
    image = Image.find(image) unless image.is_a?(Image)
    image.owner == self
  end

  def full_name
    [first_name, last_name].compact.join(' ')
  end

  # TODO all of this needs refactoring/rethinking, for thoughts see:
  # https://git-aws.internal.justin.tv/esports/tourney-edge/wiki
  def self.handle_user_change(event)
    event = Hashie::Mash.new(event)
    data = event.data

    return false if event.type != 'USER_CHANGE'

    # When a user's email changes, we update their Person to be a member
    # in all orgs for that email addresses domain, and only those orgs.
    if data.diff.present? && data.diff.keys.include?('email')
      orgs = Organization.for_email(data.email)

      # If no Organization matches, create one
      if orgs.empty?
        orgs = [Organization.create!(
          name: data.email.scan(/@(.*?)\..*$/).flatten.join.capitalize,
          email_domains: data.email.scan(/@(.*?)$/).flatten
        )]
      end

      person = Person.where(user_id: data.id).first_or_create!
      person.update!(organizations: orgs)
    end
  end
end
