class Series < ApplicationRecord
  belongs_to :game
  belongs_to :tournament_stage
  has_many :matches
  has_many :opponents, as: :contest, inverse_of: :contest
  has_many :teams, through: :opponents, source: :contender, source_type: 'Team'
  has_many :people, through: :opponents, source: :contender, source_type: 'Person'
  accepts_nested_attributes_for :matches, allow_destroy: true
  accepts_nested_attributes_for :opponents, allow_destroy: true
  validate :validate_matches
  class_attribute :max_matches

  before_validation :default_scheduled_at
  validate :validate_scheduled_at

  class Bo3 < Series
    self.max_matches = 3
  end

  class Bo5 < Series
    self.max_matches = 5
  end

  class Bo7 < Series
    self.max_matches = 7
  end

  def stage
    self.tournament_stage
  end

  def winner_id
    return nil unless self.max_matches
    wins_needed = (self.max_matches + 1) / 2
    matches.inject(Hash.new(0)) do |wins, match|
      if (id = match.winner&.contender_id) && match.state != 'contested'
        return id if (wins[id] += 1) >= wins_needed
      end
      wins
    end
    nil
  end

  def winner
    (id = winner_id) && Team.find(id)
  end

  def team_ids
    @team_ids ||= matches.collect{|m|
      m.opponents.where(contender_type: 'Team').collect(&:contender_id)
    }.flatten
  end

  def teams
    @teams ||= Team.where(id: team_ids)
  end

  private

  def validate_matches
    matches.find_all(&:invalid?).each do |match|
      errors.add(:matches, match.errors.full_messages.join(', '))
    end
  end

  def default_scheduled_at
    self.scheduled_at ||= matches.map(&:scheduled_at).compact.min
  end

  def validate_scheduled_at
    if matches.any? {|match| match.scheduled_at && scheduled_at > match.scheduled_at}
      errors.add(:scheduled_at, 'must be before first scheduled match')
    end
  end
end
