class Team < ApplicationRecord
  include FriendlyId
  include Sociable

  attribute :join_token, :string, default: -> { TeamInvitation.token }
  belongs_to :organization, required: false
  belongs_to :game, required: false
  has_many :team_members, dependent: :destroy
  has_many :people, through: :team_members
  has_many :match_teams
  has_many :match_players
  has_many :tournament_entries, as: :entrant
  has_many :tournaments, through: :tournament_entries
  has_many :team_invitations
  has_many :opponents, as: :contender, inverse_of: :contender
  validates :name, presence: true, uniqueness: true
  validate :validate_country
  validate_social :social

  mount_uploader :cover, TeamCoverUploader
  validates_integrity_of :cover
  validates_processing_of :cover
  validates_download_of :cover

  mount_uploader :logo, TeamLogoUploader
  validates_integrity_of :logo
  validates_processing_of :logo
  validates_download_of :logo

  friendly_id :name, use: :slugged

  attribute :settings, :jsonb, default: {}
  validate :validate_charity

  # settings['hidden']
  def hidden
    !!settings['hidden']
  end
  def hidden=(value)
    value.downcase! if value.is_a?(String)
    settings['hidden'] = ![false, 0, '0', 'false', 'f', 'no', 'null', 'nil', nil].include?(value)
  end
  before_validation do |article|
    if article.settings_changed? && article.settings.include?('hidden')
      # Enforce storing boolean.
      article.hidden = article.settings['hidden']
    end
  end

  # settings['charity']
  class_attribute :approved_charities
  Team.approved_charities = [
    nil,
    'Direct Relief',
    'Extra Life',
    'Game Changer',
    'Gamers Outreach',
    'Prevent Cancer Foundation'
  ]
  def charity
    settings['charity']
  end
  def charity=(new_charity)
    settings['charity'] = new_charity
  end
  def validate_charity
    unless self.class.approved_charities.include?(charity)
      errors.add(:settings, "unapproved charity: #{charity.inspect}")
    end
  end

  def validate_country
    return if country.blank?
    @country =
      case country.length
      when 2 then ISO3166::Country.find_country_by_alpha2(country)
      when 3 then ISO3166::Country.find_country_by_alpha3(country)
      else ISO3166::Country.find_country_by_name(country)
      end
    if @country.nil?
      errors.add(:country, "is invalid: #{country.inspect}")
      return
    end
    self.country = @country.name
  end

  def region
    @country&.region
  end

  def invite_email(email)
    team_invitations.create!(
      email: email
    )
  end

  def members(role = nil)
    return team_members if role.nil?
    team_members.select {|m| m.roles.include?(role)}
  end

  def can_be_modified_by?(person)
    TeamMember.can_modify?(person, self)
  end

  def add_member!(person, roles=[])
    members.create!(person: person, roles: roles)
  end

  # Array of Organizations represented in this team, ordered by membership.
  def organizations
    # Memoize so Ruby doesn't re-evaluate `memberships` on every lookup:
    counts = memberships
    people.map(&:organizations).flatten.uniq.
      sort_by!{|org|counts[org.id]}.reverse
  end

  # Mapping of `{ Organization ID => count of team members in org }`.
  def memberships
    # Hold on to your butts:
    people.map(&:organizations).flatten.
      inject(Hash.new(0)) {|orgMap,org| orgMap.update(org.id => orgMap[org.id]+1)}
  end
end
