class TeamInvitation < ApplicationRecord
  belongs_to :team
  attribute :token, :string, default: -> { TeamInvitation.token }
  validates :token, uniqueness: true
  validates :email,
    uniqueness: { scope: [:team_id, :used] },
    format: { with: /\A[\w+\-.]+@[a-z\d\-]+(\.[a-z\d\-]+)*\.[a-z]+\z/i }

  default_scope { where(used: false) }

  # Creates a new token for invitations. Also used by Team#join_token
  def self.token
    SecureRandom.hex(4)
  end

  # Accepts the invitation on behalf of the given user_id
  def accept(person)
    errors.add(:user, "need user to accept invitation for") and return false unless person&.valid?
    errors.add(:used, 'invitation has been used') and return false if used?
    team.add_member!(person)
    update(used: true, user_id: person.user_id)
  rescue => e
    errors.add(:team, "can't add this user to the team: #{e.message}")
    return false
  end

  def accept!(person)
    if !accept(person) || errors.any?
      raise ActiveRecord::RecordInvalid.new(self)
    end
  end
end
