class Tournament < ApplicationRecord
  include AASM
  include FriendlyId

  belongs_to :game, required: false
  belongs_to :season, required: false
  has_many :tournament_entries
  has_many :teams, through: :tournament_entries, source: :entrant, source_type: 'Team'
  has_many :stages, class_name: TournamentStage, inverse_of: :tournament
  accepts_nested_attributes_for :stages

  mount_uploader :cover, TournamentCoverUploader
  validates_integrity_of :cover
  validates_processing_of :cover
  validates_download_of :cover

  mount_uploader :logo, TournamentLogoUploader
  validates_integrity_of :logo
  validates_processing_of :logo
  validates_download_of :logo

  friendly_id :qualified_name, use: :slugged
  def qualified_name
    # Tournament names end up being things like "NA Qualifier".
    # Including League and Season makes it descriptive and unique:
    # "RLCS" (League) + "Season 3" (Season) + "NA Qualifier" (Tournament)
    [season&.league&.name, season&.name, self.name].compact.join(' ')
  end

  validates_associated :tournament_entries
  validates :name, presence: true
  validates :entrants_type, presence: true

  # `#destroy` returns the last state of a Tournament, and that includes its stages' standings,
  # but determining those depends on this Tournament being non-`nil`.
  # So we need to force evaluation & memoization of those standings before destroying self.
  before_destroy do |record|
    record.stages.each {|stage| stage.standings if stage.respond_to?(:standings)}
  end

  scope :ongoing, -> { where.not(aasm_state: :done) }

  aasm(whiny_transactions: true) do
    state :created, initial: true
    state :published,
      :registration_open,
      :registration_closed,
      :started,
      :finals,
      :done

    event :reset do
      transitions to: :created
    end

    event :open_registration do
      transitions to: :registration_open
    end

    event :close_registration do
      transitions to: :registration_closed
    end
  end

  def state; read_attribute(:aasm_state); end

  def enter_team!(team)
    tournament_entries.create!(
      entrant_type: Team,
      entrant_id: team.id
    )
  end

  def starts_at
    return nil if stages.empty?
    stages.collect(&:starts_at).min
  end

  def ends_at
    return nil if stages.empty?
    stages.collect(&:ends_at).max
  end
end
