class TournamentStage < ApplicationRecord
  belongs_to :tournament, inverse_of: :stages
  validates :starts_at, presence: true
  validates :ends_at, presence: true
  validates :color, color: true

  default_scope -> { order(:starts_at) }

  def progress
    @progress = (starts_at - Time.now) / (starts_at - ends_at)
    [[0.0, @progress].max, 1.0].min
  end

  def elimination?
    false
  end

  class RoundRobin < TournamentStage
    has_many :series, foreign_key: :tournament_stage_id

    def standings
      return @standings if @standings
      return [] if tournament.nil?

      teams = tournament.teams.reject(&:hidden)
      return [] if teams.empty?

      # Pre-populate with all entrants.
      scores = Hash[teams.map do |team|
        [team.id, {
          contender: {schema: 'team', id: team.id, name: team.name},
          wins: 0, losses: 0, match_wins: 0, match_losses: 0,
        }]
      end]
      scores.default = Hash.new(0) # Discard win/loss info for hidden teams.

      series.each do |series|
        series.matches.each do |match|
          if winner = match.winner&.contender
            scores[winner.id][:match_wins] += 1
            match.opponents.map(&:contender_id).each do |loser_id|
              scores[loser_id][:match_losses] += 1 unless loser_id == winner.id
            end
          end
        end

        if winner = series.winner
          scores[winner.id][:wins] += 1
          series.opponents.map(&:contender_id).each do |loser_id|
            scores[loser_id][:losses] += 1 unless loser_id == winner.id
          end
        end
      end

      @standings = scores.values.sort_by do |score|
        series_played = score[:wins] + score[:losses]
        series_win_pct = series_played == 0 ? 0.0 : (score[:wins].to_f / series_played)
        matches_played = score[:match_wins] + score[:match_losses]
        match_win_pct = matches_played == 0 ? 0.0 : (score[:match_wins].to_f / matches_played)
        [
          # If at some point we revert to pushing 0-0's to the end, uncomment these two:
          # series_played > 0 ? 0 : 1,
          # matches_played > 0 ? 0 : 1,
          -score[:wins],
          score[:losses],
          -match_win_pct,
          score[:contender][:name] ]
      end
    end
  end

  class SingleElim < TournamentStage
    has_many :series, foreign_key: :tournament_stage_id

    def elimination?
      true
    end
  end

  class DoubleElim < TournamentStage
    has_many :series, foreign_key: :tournament_stage_id

    def elimination?
      true
    end
  end
end
