class Curse
  class CurseAPIError < StandardError; end

  CURSE_HOST = 'curseapp.net'

  def initialize(server_id=nil, auth_token=nil)
    # TODO: Get server/auth info per client; see https://trello.com/c/YL1hfebI
    @server_id = server_id || ENV['CURSE_SERVER']
    @auth_token = auth_token || ENV['CURSE_AUTH_TOKEN']
    if @server_id.nil? || @auth_token.nil?
      raise CurseAPIError.new('missing Curse server ID and/or auth token')
    end
  end

  def channels
    server_info['Channels'].map {|c| Channel.new(self, c)}
  end

  # Returns the mapping `{ folder name => { id: uuid, channels: [...] } }`.
  # Note that `nil` is a valid folder name.
  def folders
    folders = {}
    channels.each do |channel|
      folder = folders.fetch(channel.folder, {
        id: channel.json['DisplayCategoryID'],
        channels: []
      })
      folder[:channels].push(channel)
      folders[channel.folder] = folder
    end
    folders
  end

  # `channel` tries to find a channel by name or id, returning `nil` if it doesn't exist.
  # Given `create: true`, will instead try to create it if it doesn't exist.
  # If using `create: true`, channel name is _required_, and a topic is optional.
  def channel(options)
    raise ArgumentError.new('need id or name to find channel') unless options[:id] || options[:name]
    raise ArgumentError.new('need name to make new channel') if options[:create] && !options.include?(:name)
    filter = options[:id] ? proc {|c| c.id == options[:id]} : proc {|c| c.name == options[:name]}
    channel = channels.detect(&filter)
    return channel unless channel.nil? && options[:create]
    make_channel(options.slice(:name, :topic, :folder))
  end

  def delete_channel(id)
    request(:delete, 'groups', "/servers/#{@server_id}/channels/#{id}")
    clear_cache
  end

  def invite(id)
    request(:post, 'groups', "/servers/#{@server_id}/invites", body: {"ChannelID" => id})['InviteUrl']
  end

  # This should really be private, but Channel needs to call it, and it shouldn't be
  # in `Curse::Channel` because it's the `Curse` class that should be making the actual API calls. :/
  def update_channel(id, settings)
    request(:post, 'groups', "/servers/#{@server_id}/channels/#{id}/settings", body: settings)
    clear_cache
  end

  def clear_cache
    @server_info = nil
  end

  private

  def conn
    @conn ||= Faraday.new do |c|
      c.adapter Faraday.default_adapter
      c.headers['AuthenticationToken'] = @auth_token
      c.headers['Content-Type'] = 'application/json'
      c.headers['Accept'] = 'application/json'
    end
  end

  def request(method, api, endpoint, body: nil)
    body = body.to_json unless body.nil?
    resp = conn.send(method.to_s.downcase.to_sym, make_url(api, endpoint), body)
    return resp.body.blank? ? nil : JSON.parse(resp.body) if resp.success?

    # Error replies from Curse API are very inconsistent.
    # Seen so far:
    # - `{"error": "..."}`
    # - `{"Message": "..."}`
    # - HTML (even with "Accept: application/json")
    if resp.headers['Content-Type'] =~ /application\/json/
      error = JSON.parse(resp.body).values_at('error', 'Message').compact[0]
    else
      error = resp.body
    end
    msg = "Curse API: #{method.to_s.upcase} #{endpoint} replied with #{resp.status}: #{error}"
    raise CurseAPIError.new(msg)
  end

  # Cached server metadata; cleared on any update.
  def server_info
    @server_info ||= request(:get, 'groups', "/groups/#{@server_id}")
  end

  def make_channel(name:, topic: nil, folder: nil)
    # Curse API sucks: folder can only be set during creation, topic can only be set after.
    body = {
      # This seems to be the minimal set of required keys:
      "Title" => name,
      "AccessRoles" => [],
      "IsPublic" => true
      # (Can't set channel topic during creation.)
    }

    if !folder.nil?
      # Curse API sucks: the only way to use folders _at all_ is with channel metadata,
      #                  i.e. can't list folders, create folders, delete folders, etc.
      # Thankfully, creating a folder by simply starting to use a new UUID seems to suffice.
      body['DisplayCategory'] = {
        'ID' => folders.dig(folder, :id) || SecureRandom.uuid,
        'Name' => folder,
        'Rank' => 1
      }
    end

    json = request(:post, 'groups', "/servers/#{@server_id}/channels", body: body)
    channel = Channel.new(self, json)
    topic.blank? ? channel : channel.set_topic(topic)
    clear_cache
    channel
  end

  def make_url(api, endpoint)
    "https://#{api}-v1.#{CURSE_HOST}#{endpoint}"
  end
end
