class Curse
  class Channel
    # Curse API is ridiculously bad when it comes to channel settings/properties (like channel topics):
    #   - can't set topic during creation; have to modify it with another request after creation
    #   - API has no PATCH for channel settings, only POST, so we have to include all settings in change request
    #   - API has no GET for a specific channel's settings
    #   - easiest way to get channel settings is to derive them from channel metadata,
    #     and even that is not a subset: field names different, AccessRoles is impossible to get
    #   - API does not provide GET for metadata for a specific channel, only for _all_ channels
    #   - response after updating settings has no body, so we have to fetch Channel metadata afterward
    #
    # So instead of just "PATCH /server/${server_id}/channels/${channel_id}" with just topic change,
    # we have to:
    #   1. derive channel settings from channel metadata (`@json`)
    #   2. modify it
    #   3. POST it
    #   4. fetch full channel list
    #   5. find own metadata in the result
    #   6. re-instantiate self with that metadata
    SETTINGS_KEYS = Set[
      'AllowTemporaryChildGroups',
      'ForcePushToTalk',
      'GroupTitle', # updating requires renaming this to "Title"
      'HideCallMembersNoAccess',
      'HideNoAccess',
      'IsPublic',
      'MessageOfTheDay'
      # updates also require 'AccessRoles', but no API _reply_ ever includes this
    ]

    {
      :id      => 'GroupID',
      :name    => 'GroupTitle',
      :topic   => 'MessageOfTheDay',
      :server  => 'RootGroupID',
      :url     => proc {|json| 'https://app.twitch.tv'+json['UrlPath']},
      :folder  => 'DisplayCategory'
    }.each do |field, val|
      if val.is_a?(String)
        define_method(field) { @json[val] }
      else
        define_method(field) { val.call(@json) }
      end
    end

    def initialize(service, json)
      @service = service
      @json = json
    end
    attr_reader :json

    def self.create(service, options)
      service.channel(options.merge({create: True}))
    end

    def delete
      @service.delete_channel(id)
      self.freeze
    end

    def invite
      @service.invite(id)
    end

    def reload
      @json = @service.channel(id: id).json
      self
    end

    def settings
      settings = @json.slice(*SETTINGS_KEYS)
      settings['Title'] = settings.delete('GroupTitle')
      settings['AccessRoles'] = []
      settings
    end

    def set_topic(new_topic)
      return self if topic == new_topic
      new_settings = settings.merge({'MessageOfTheDay' => new_topic})
      @service.update_channel(id, new_settings)
      reload
    end

    def to_s
      "Channel(" \
        + [:name, :id, :url, :server].map do |sym|
            "#{sym.to_s}=#{self.send(sym).inspect}"
          end.join(', ') \
        + ")"
    end
  end
end
