# Imports a smash.gg tournament to EDB.
#
# The mapping of smashgg models to EDB's is:
#           EDB: League <   Season   <   Tournament   <     Stage     < Series < Match
#      Smash.gg:  (n/a) < Tournament <     Event      < Phase < Group <  Set   < Game
# example names: "RLCS" < "Season 3" < "NA Qualifier" < "League Play"

require 'json'
require 'net/http'

class Smashgg
  def log(level='INFO', msg)
    msg = "#{level} [#{Time.now.to_s}] " + msg
    puts msg
    @log.puts(msg)
  end

  def http_get(target)
    uri = URI(target)
    Net::HTTP.start(uri.host, uri.port, use_ssl: uri.scheme == 'https') do |http|
      req = Net::HTTP::Get.new(uri)
      resp = http.request(req)
      status = resp.code.to_i # areyoufuckingkiddingme.jpg

      return [status, resp.body] unless status == 200
      if status == 200 && block_given? && (block_result = yield(resp))
        return [500, block_result]
      end
      [200, resp.body]
    end
  end

  def http_get_json(model, id)
    if @cache
      path = File.join(@cache, "#{model}-#{id}.json")
      if File.exists?(path)
        log("[http_get_json] cache hit: #{path.inspect}")
        return JSON.load(open(path))['entities']
      end
    end

    url = 'https://api.smash.gg/' +
      case model
      when 'tournament' then "tournament/#{id}?expand[]=event&expand[]=phase&expand[]=groups"
      when 'group' then "phase_group/#{id}?expand[]=sets&expand[]=entrants"
      else raise 'no'
      end
    log("[http_get_json] downloading #{model} #{id}: #{url}")
    dl_start = Time.now
    status, body = http_get(url) do |resp|
      unless resp.content_type() == 'application/json'
        "got #{resp.content_type.inspect} instead of JSON"
      end
    end
    log("[http_get_json] download of #{model} #{id} took #{(Time.now - dl_start).round(3)}s")
    if status != 200
      log('ERROR', "[http_get_json] failed to get #{model} #{id.inspect} from Smash.gg API: #{body.inspect}")
      return nil
    end

    File.open(path, 'w') {|f| f.binmode.write(body)} if @cache

    JSON.load(body)['entities']
  end

  def http_get_file(target, name_pat)
    # Apparently a neat Ruby trick: `Proc.new` with no args refers to block passed
    # to the current method _only if_ one was passed, and nil otherwise.
    # It's meant for exactly the use case here: forwarding an optional block to another method.
    status, body = http_get(target, &Proc.new)
    return [status, body] if status != 200
    Tempfile.open(name_pat) do |tmp|
      tmp.binmode.write(body)
      [200, tmp.path]
    end
  end

  def http_get_image(url, log_prefix='')
    log("#{log_prefix}downloading image: #{url}")
    status, result = http_get_file(url, 'smashgg-import_') do |resp|
      unless resp.content_type.start_with?('image/')
        "wrong content type: #{resp.content_type.inspect}"
      end
    end

    if status != 200
      log('ERROR', "#{log_prefix}error fetching image #{url.inspect}: #{result.inspect}")
      return nil
    end

    path = result + File.extname(url)
    File.rename(result, path)
    log("#{log_prefix}saved to: #{path}")
    return path
  end

  # Default filter for events and phases.
  # Must apply to both because Smashgg API is inconsistent with where these names occur,
  # e.g. in RLCS3 there are generically-named events, and those contain phases whose names match /open qualifier/i,
  #      whereas in RLCS4 there are _events_ that match /open qualifier/i whose phases are something generic like "Bracket".
  def skip?(name)
    name =~ /(open|make-up) qualifier/i || name =~ /play-in/i || name == 'DQ'
  end

  def initialize
    # Initialize logging.
    log_base = Pathname.new(ENV.fetch('SMASHGG_LOG', Rails.root.join('log', 'smashgg')))
    @log = File.open(log_base.sub_ext('.log'), 'w')
    ActiveRecord::Base.logger&.reopen(log_base.sub_ext('-ar.log'))

    # Initialize cache.
    @cache = ENV.fetch('SMASHGG_CACHE', Rails.root.join('tmp', 'cache', 'smashgg').to_s)
    @cache = nil if @cache.blank? # to support disabling cache with `SMASHGG_CACHE=`
    FileUtils.mkdir_p(@cache) if @cache
  end

  def make_all_tournaments(smashgg_tournament_id)
    # Fetch JSON.
    j = http_get_json('tournament', smashgg_tournament_id)
    return if j.nil?

    # League & Season
    name = j['tournament']['name']
    unless name =~ /(.*) (season [0-9]+)/i
      log('ERROR', "[make_all_tournaments] failed to extract League and Season names from #{name.inspect}")
      return false
    end
    league_name = $1
    season_name = $2
    if @league = League.find_by(name: league_name)
      log("[make_all_tournaments] found existing League #{league_name.inspect}: #{@league.id}")
    else
      @league = League.create!(name: league_name)
      log("[make_all_tournaments] created League #{league_name.inspect}: #{@league.id}")
    end
    if @season = Season.find_by(name: season_name, league: @league)
      log("[make_all_tournaments] found existing Season #{season_name.inspect}: #{@season.id}")
    else
      @season = Season.create!(name: season_name, league: @league)
      log("[make_all_tournaments] created Season #{season_name.inspect}: #{@season.id}")
    end

    # Game
    make_game(j['videogame'][0])

    # Some pre-processing for convenience:
    # - embed groups within their phases, and phases within their events.
    # - sort phases by "phaseOrder"
    # - remove events and phases that would be skipped (see `skip?`)
    j['phase'].reject! {|j_phase| skip?(j_phase['name'])}
    j['phase'].each do |j_phase|
      j_phase['groups'] = j['groups'].select do |j_group|
        j_group['phaseId'] == j_phase['id']
      end
    end
    j['event'].each do |j_event|
      j_event['phases'] = j['phase'].select do |j_phase|
        j_phase['eventId'] == j_event['id']
      end.sort_by {|j_phase| j_phase['phaseOrder']}
    end
    j['event'].reject! {|j_event| skip?(j_event['name']) || j_event['phases'].empty?}

    # Initialize teams mapping.
    # Missing keys here mean that the API response refers to an entrant it never defined.
    # This is always an unrecoverable error, so raising on a miss saves us a lot of boilerplate.
    @teams = Hash.new {|h,entrantId| raise "error resolving entrantId #{entrantId.inspect} to a Team"}

    # Create Tournaments out of events.
    j['event'].each do |j_event|
      make_tournament(j_event)
    end

    log("[make_all_tournaments] finished")
    @log.flush
  end

  def make_game(j_game)
    @game = Game.find_by(name: j_game['displayName'])
    unless @game.nil?
      log("[make_game] found existing Game #{j_game['displayName'].inspect}: #{@game.id}")
      return
    end

    if j_image = j_game['images'].select{|img| img['isOriginal']}[0]
      img_path = http_get_image(j_image['url'], "[make_game][#{j_game['displayName']}] ")
      image = img_path ? open(img_path) : nil
    end

    @game = Game.create!(name: j_game['displayName'], abbr: j_game['abbrev'], cover: image)
    log("[make_game] created Game #{@game.name.inspect} (#{@game.abbr.inspect}): #{@game.id}")
    File.unlink(img_path) if image
  end

  def make_tournament(j_event)
    name = j_event['name']
    if tournament = Tournament.find_by(season: @season, name: name)
      log("[make_tournament] found existing Tournament #{name.inspect}: #{tournament.id}")
    else
      tournament = Tournament.create!(
        name: name,
        game: @game,
        season: @season,
        entrants_type: 'Team',
        aasm_state: :registration_open,
      )
      log("[make_tournament] created Tournament #{name.inspect}: #{tournament.id}")
    end

    # Create TournamentStages out of phases.
    make_stages(tournament, j_event)
  end

  STAGE_TYPES = {
    1 => 'TournamentStage::SingleElim',
    2 => 'TournamentStage::DoubleElim',
    3 => 'TournamentStage::RoundRobin',
    # 4 => "Swiss"
    # 5 => (undefined)
    # 6 => "Custom Schedule"
    # 7 => "Matchmaking"
  }

  def make_stages(tournament, j_event)
    log("[make_stages] found #{j_event['phases'].length} phases")
    j_event['phases'].each do |j_phase|
      j_phase['groups'].each do |j_group|
        name = j_phase['groups'].length == 1 ? j_phase['name'] : "#{j_phase['name']} - #{j_group['identifier']}"
        if stage = TournamentStage.find_by(tournament: tournament, name: name)
          log("[make_stages] found existing TournamentStage #{name.inspect}: #{stage.id}")
        elsif !STAGE_TYPES.include?(j_group['groupTypeId'])
          log("[make_stages] skipping unsupported groupTypeId: #{j_group['groupTypeId'].inspect}")
          next
        else
          stage = TournamentStage.create!(
            tournament: tournament,
            name: name,
            type: STAGE_TYPES[j_group['groupTypeId']],
            starts_at: Time.at(j_event['startAt']),
            ends_at: Time.at(j_event['endAt']),
            color: '#000000', # default doesn't work?
          )
          log("[make_stages] created #{stage.type} #{name.inspect}: #{stage.id}")
        end

        process_group(j_group['id'], stage)
      end
    end
  end

  def process_group(id, stage)
    j = http_get_json('group', id)
    return if j.nil?

    if j['entrants']
      # Smashgg has given us null for this before; don't crash in this case.
      make_teams(j['entrants'])
      make_tournament_entries(j['entrants'], stage.tournament)
    end
    make_series(j['sets'], stage) if j['sets']
  end

  def make_teams(j_entrants)
    j_entrants.each do |j_entrant|
      entrantId = j_entrant['id']
      eventId = j_entrant['eventId']
      team_name = j_entrant['name']
      if team = Team.find_by("metadata->'smashgg'->'entrantIds'->>? = ?", eventId.to_s, entrantId.to_s)
        # Check for updates (currently, only name):
        if team.name != team_name
          log("[make_teams] updating name for Team #{team.id}: #{team.name.inspect} -> #{team_name.inspect}")
          team.update!(name: team_name)
        end
      elsif team = Team.find_by(name: team_name)
        # Check for existing team by name. If they don't have smashgg metadata, tag it.
        if (cur_id = team.metadata.dig('smashgg', 'entrantIds', eventId)).blank?
          log("[make_teams] tagging existing team #{team.name.inspect} (#{team.id}) " \
              + "with (eventId, entrantId) = (#{eventId}, #{entrantId})")
          team.metadata['smashgg'] = {} unless team.metadata.include?('smashgg')
          team.metadata['smashgg']['entrantIds'] = {} unless team.metadata['smashgg'].include?('entrantIds')
          team.metadata['smashgg']['entrantIds'][eventId] = entrantId
          team.save!
        else
          # It has a Smashgg ID, and it's not this one. Something has gone horribly wrong.
          log('ERROR', "[make_teams] Team #{team.name} (#{team.id}) has unexpected entrantId: " \
                     + "expected #{entrantId}, found #{cur_id}")
          # Unfortunately, this means we can't continue -- the rest of the response probably uses this
          # new `entrantId`, but we don't know if it's this team or a different one with the same name.
          raise "error resolving Team with entrantId = #{entrantId}"
        end
      end

      if team.nil?
        team = Team.create!(
          name: team_name,
          game: @game,
          metadata: {'smashgg' => {'entrantIds' => {eventId => entrantId}}},
        )
        log("[make_teams] created Team #{team_name.inspect}: #{team.id}")
      end
      @teams[j_entrant['id']] = team

      # Create/update members.
      expected_members = [] # Person IDs for the team members Smashgg says this team has.
      j_entrant['mutations']['participants'].values.each do |j_participant|
        handle = [j_participant['prefix'], j_participant['gamerTag']].select{|s|!s.blank?}.join(' ')
        person = Person.find_by("metadata->'smashgg'->>'playerId' = ?", j_participant['playerId'].to_s) \
              || Person.find_by(handle: handle)
        if person
          # Check smashgg tag.
          if person.metadata.dig('smashgg', 'playerId').blank?
            log("[make_teams] tagging existing Person #{person.id} with playerId = #{j_participant['playerId']}")
            person.metadata['smashgg'] = {} unless person.metadata.include?('smashgg')
            person.metadata['smashgg']['playerId'] = j_participant['playerId']
            person.save!
          elsif person.metadata['smashgg']['playerId'] != j_participant['playerId']
            log('ERROR', "[make_teams] Person #{person.handle} (#{person.id}) has unexpected playerId: " \
                + "expected #{j_participant['playerId']}, found #{person.metadata['smashgg']['playerId']}")
            # Team roster is the only place a participant is referred to; safe to continue with other changes.
            next
          end

          # Check for updates: first_name, last_name, handle
          edb_names = person.attributes.slice('first_name', 'last_name', 'handle').symbolize_keys
          smashgg_names = {
            first_name: j_participant.dig('contactInfo', 'nameFirst'),
            last_name: j_participant.dig('contactInfo', 'nameLast'),
            handle: handle,
          }.reject {|_, name| name.nil?}
          diff = Hash[smashgg_names.to_a - edb_names.to_a]
          unless diff.empty?
            log("[make_teams] name change for Person #{person.id}: " \
                + diff.keys.map {|field| "#{field}: #{edb_names[field].inspect} -> #{smashgg_names[field].inspect}"}.join(', '))
            person.update!(diff)
          end

        else
          person = Person.create!(
            handle: handle,
            first_name: j_participant.dig('contactInfo', 'nameFirst') || '',
            last_name: j_participant.dig('contactInfo', 'nameLast') || '',
            user_id: SecureRandom.uuid,
            metadata: {'smashgg' => {'playerId' => j_participant['playerId']}},
          )
          log("[make_teams] created Person #{handle.inspect}: #{person.id}")
        end

        expected_members.push(person.id)
      end

      # Check/update team membership.
      current_members = team.person_ids.sort
      expected_members.sort!
      if !(new_members = expected_members - current_members).empty?
        log("[make_teams] adding #{new_members.length} new people to Team #{team.name}: #{new_members.join(', ')}")
        team.person_ids += new_members
      end
      if !(old_members = current_members - expected_members).empty?
        log("[make_teams] removing #{old_members.length} people from Team #{team.name}: #{old_members.join(', ')}")
        team.person_ids -= old_members
      end
    end
  end

  def make_tournament_entries(j_entrants, tournament)
    already_exist = 0
    j_entrants.each do |j_entrant|
      team = @teams[j_entrant['id']]
      if TournamentEntry.exists?(tournament: tournament, entrant: team)
        already_exist += 1
      else
        # This isn't the normal AHGL flow; skip the "open registration" requirement for making a new entry.
        entry = TournamentEntry.create!(tournament: tournament, entrant: team)
        entry.save!(validate: false)
        log("[make_tournament_entries] registered Team #{team.name} for Tournament #{tournament.name}")
      end
    end
    log("[make_tournament_entries] #{already_exist} Teams are already registered for Tournament #{tournament.name}") if already_exist > 0
  end

  def add_elim_links(elim_links, j_set)
    return unless [1,2].all? {|i| j_set["entrant#{i}PrereqType"] == 'set'}
    [1,2].map do |i|
      prev = j_set["entrant#{i}PrereqId"]
      link = elim_links.fetch(prev, {winner_to: nil, loser_to: nil})
      link["#{j_set["entrant#{i}PrereqCondition"]}_to".to_sym] = j_set['id']
      elim_links[prev] = link
    end
    elim_links
  end

  def make_series(j_sets, stage)
    log_base_prefix = "[make_series][stage = #{stage.name.inspect}]"

    # Sanity checks that should prevent creation of even a stub or forged Series altogether:
    i_set, num_sets = 0, j_sets.length
    no_start_count = 0
    nil_entrants_count = 0
    bye_count = 0
    j_sets.select! do |j_set|
      i_set += 1
      log_prefix = log_base_prefix + "[set #{i_set}/#{num_sets}, id=#{j_set['id']}] "
      startAt = j_set['startAt'] || j_set['startedAt'] || j_set['completedAt']
      # Reject sets with either entrant null, unless it's an elimination stage, in which case we _do_ want placeholders.
      if (entrant_ids = j_set.values_at('entrant1Id', 'entrant2Id')).any?(&:blank?) && !stage.elimination?
        nil_entrants_count += 1
        false
      # However, we do still want to filter out these extraneous placeholders that Smashgg loves so much. x_x
      elsif stage.elimination? && !j_set.values_at(*[1,2].map{|i|"entrant#{i}PrereqType"}).all?{|c|['seed','set'].include?(c)}
        bye_count += 1
        false
      # Also filter out Series with no start time, _unless_ they're a disqualified set or a placeholder in an elimination bracket.
      elsif startAt.blank? && !stage.elimination? && !j_set.values_at('entrant1Score', 'entrant2Score').include?(-1)
        no_start_count += 1
        false
      else
        true
      end
    end
    log(log_base_prefix + " skipping #{nil_entrants_count} series with one or both entrants nil") if nil_entrants_count > 0
    log(log_base_prefix + " skipping #{no_start_count} series with no start time") if no_start_count > 0
    log(log_base_prefix + " skipping #{bye_count} dummy bye series") if bye_count > 0

    # Throughout Series creation/updates, this is { Series ID => {winner_to: setId, loser_to: setId} }
    # These setIds will be converted to Series IDs after we're sure they all exist.
    elim_links = {}       # { setId => {winner_to: setId, loser_to: setId} }
    series_of_setId = {}  # { setId => Series }

    j_sets.each_with_index do |j_set, i_set|
      log_prefix = log_base_prefix + "[set #{i_set+1}/#{j_sets.length}, id=#{j_set['id']}] "
      if series = Series.find_by("metadata->'smashgg'->>'setId' = ?", j_set['id'].to_s)
        update_series(series, j_set)
        series_of_setId[j_set['id']] = series
        add_elim_links(elim_links, j_set)
        next
      end

      # New Series.
      # startAt: When a contender is disqualified, the API response has null "startAt" and "startedAt".
      startAt = j_set.values_at('startAt', 'startedAt', 'completedAt').compact.first
      startAt = Time.at(startAt) if startAt
      team_names = j_set.values_at('entrant1Id', 'entrant2Id').map {|eid| @teams.fetch(eid, nil)&.name || '(TBD)'}
      log(log_prefix + "creating new Series: #{team_names.join(' vs ')} at #{startAt&.to_s || '(TBD)'}")

      metadata = {
        'smashgg' => {
          'setId'   => j_set['id'],
          'eventId' => j_set['eventId'],
          'groupId' => j_set['phaseGroupId'],
        },
        'disqualified' => j_set.values_at('entrant1Score', 'entrant2Score').include?(-1),
      }
      if stage.elimination?
        metadata.merge!({
          'ident' => j_set['identifier'],
          'round' => j_set['round'],
        })
        add_elim_links(elim_links, j_set)
      end

      series = stage.series.build(
        game: @game,
        type: "Series::Bo#{j_set['bestOf']}",
        scheduled_at: startAt,
        metadata: metadata,
      )
      [1,2].each do |i|
        next unless id = j_set["entrant#{i}Id"]
        score = j_set["entrant#{i}Score"].to_i
        opp = series.opponents.build(
          contender: @teams[id],
          score: [score, 0].max, # API could have -1 to signify DQ
          metadata: {
            'smashgg' => series.metadata['smashgg'].merge({
              'entrantIndex' => i,
              'entrantId'    => id,
            }),
            'disqualified' => score == -1,
          },
        )
      end

      # There are some special cases to worry about when "games" is an empty array:
      # 1. scores and "completedAt" are all null -> The series hasn't played out yet. Create a new Series with 1 Match and no winner.
      # 2. one (or both?) scores -1 -> One (or both?) contenders were disqualified. Forge a clean sweep.
      # 3. "completedAt" is non-null and both scores > 0 -> Missing data. Forge matches to show a reverse sweep.
      # (TODO: Promote scores/winners to Series model and make Matches optional.)
      if j_set['games'].empty?
        _, score1, score2 = triple = j_set.values_at('completedAt', 'entrant1Score', 'entrant2Score')

        # Special case 1: Series hasn't happened yet.
        if triple.all?(&:blank?)
          log(log_prefix + "Series hasn't started; Series will have no Matches")

        # Special case 2: Disqualification.
        elsif [score1, score2].include?(-1)
          log(log_prefix + "found a score of -1; creating DQ Series")
          j_set['games'] = [{
            'entrant1Id' => j_set['entrant1Id'],
            'entrant2Id' => j_set['entrant2Id'],
            'entrant1P1Stocks' => score1 == -1 ? 0 : 1,
            'entrant2P1Stocks' => score2 == -1 ? 0 : 1,
            'setId' => j_set['id'],
            'winnerId' => j_set['winnerId'],
          }] * (j_set['bestOf'] / 2 + 1)

        # Special case 3: Missing data.
        elsif !triple.any?(&:blank?)
          winner = j_set['entrant1Id'] == j_set['winnerId'] ? 1 : 2
          loser = 3 - winner
          log(log_prefix + "no match data; forging #{j_set["entrant#{loser}Score"]} loser wins + #{j_set["entrant#{winner}Score"]} winner wins")
          # Loser wins all their games first:
          j_set["entrant#{loser}Score"].times do
            j_set['games'].push({
              'entrant1Id' => j_set['entrant1Id'],
              'entrant2Id' => j_set['entrant2Id'],
              "entrant#{loser}P1Stocks" => 1,
              "entrant#{winner}P1Stocks" => 0,
              'setId' => j_set['id'],
              "winnerId" => j_set["entrant#{loser}Id"],
            })
          end
          # Then winners pull a reverse sweep:
          j_set["entrant#{winner}Score"].times do
            j_set['games'].push({
              'entrant1Id' => j_set['entrant1Id'],
              'entrant2Id' => j_set['entrant2Id'],
              "entrant#{winner}P1Stocks" => 1,
              "entrant#{loser}P1Stocks" => 0,
              'setId' => j_set['id'],
              "winnerId" => j_set["entrant#{winner}Id"],
            })
          end

        # Bad data.
        else
          log(log_prefix + "bad data for series: #{j_set.slice('completedAt', 'entrant1Score', 'entrant2Score').inspect}")
          next
        end
      end

      # Create Matches.
      j_set['games'].each_with_index do |j_game, i_game|
        match = series.matches.build(
          game: @game,
          scheduled_at: series.scheduled_at,
          metadata: {'smashgg' => {
            'gameId'     => j_game['id'],
            'setId'      => j_game['setId'],
          }},
        )
        [1,2].each do |i|
          # Very similar to creation of Series opponents,
          # except order of entrants may have changed and score is tracked differently.
          next unless id = j_game["entrant#{i}Id"]
          score = j_game["entrant#{i}P1Stocks"].to_i
          opp = match.opponents.build(
            contender: @teams[id],
            score: score,
            metadata: {
              'smashgg' => match.metadata['smashgg'].merge({
                'entrantIndex' => i,
                'entrantId'    => id,
              }),
              'disqualified' => j_set["entrant#{i}Score"] == -1,
            },
          )
          opp.winner = opp if j_game['winnerId'] == id
        end
      end
      series.save!
      series_of_setId[j_set['id']] = series
    end

    # Post-processing for single- and double-elimination stages:
    if stage.type.end_with?('Elim')
      log(log_base_prefix + " setting winner_to/loser_to references")
      elim_links.each do |setId, meta|
        next unless series = series_of_setId[setId]
        meta.transform_values! {|setId| series_of_setId[setId]&.id}
        series.metadata.merge!(meta)
        if series.changed?
          log(log_base_prefix + "[Series #{series.id}] updated winner_to=#{meta[:winner_to].inspect}, loser_to=#{meta[:loser_to].inspect}")
          series.save!
        end
      end
    end

    log(log_base_prefix + " finished creating series")
  end

  def update_series(series, j_set)
    log_prefix = "[update_series][set #{j_set['id']}][Series #{series.id}] "

    # Update disqualification status.
    # There are different updates to make when a Series _becomes_ disqualified for the first time,
    # hence `{was,is,became}_disqualified`.
    score1, score2 = j_set.values_at('entrant1Score', 'entrant2Score').map(&:to_i)
    was_disqualified = series.metadata.fetch('disqualified', false)
    is_disqualified = series.metadata['disqualified'] = [score1, score2].include?(-1)
    became_disqualified = !was_disqualified && is_disqualified
    log(log_prefix + "a contender was disqualified") if became_disqualified

    # Save changes made to metadata, if any.
    series.save! if series.changed?

    # Check if Opponents scheduled to compete have changed.
    update_opponents(series, j_set)

    # Bear in mind Opponents can have anywhere from 0 to 2 elements.
    opponents = series.opponents.to_a
    opponents.reverse! if j_set['entrant2Id'] && opponents[1]&.contender_id != @teams[j_set['entrant2Id']].id
    opp1, opp2 = opponents # either can be nil

    # Update Opponents' metadata (currently only disqualification status).
    opponents.each_with_index do |opponent, i|
      opponent.metadata['disqualified'] = j_set["entrant#{i+1}Score"] == -1
      opponent.save! if opponent.changed?
    end

    # Check for updates to scores.
    # (Dumb detail: score{1,2} is -1 for a DQ, but EDB stores those differently.
    edb_score1, edb_score2 = [score1,0].max, [score2,0].max
    if [opp1&.score.to_i, opp2&.score.to_i] != [edb_score1, edb_score2]
      log(log_prefix + "updating scores: #{opp1&.score.to_i} to #{opp2&.score.to_i} -> #{edb_score1} to #{edb_score2}")
      opp1.update!(score: edb_score1) if opp1 && opp1.score != edb_score1
      opp2.update!(score: edb_score2) if opp2 && opp2.score != edb_score2
    end

    # Check for updates in scheduled time.
    # Series and Matches were ideally created based on "startAt", but the jokers at Smashgg love to randomly destroy information,
    # so we leave out updating based on "startedAt" or "completedAt-durationSeconds".
    # And disqualification removes both "startAt" and "startedAt", so in that case just don't update whatever we have.
    unless is_disqualified || j_set['startAt'].blank?
      startAt = Time.at(j_set['startAt'])
      if series.scheduled_at != startAt
        log(log_prefix + "updating scheduled time: (#{series.scheduled_at.to_s}) -> (#{startAt.to_s})")
        # Validation to worry about: Series must start before earliest Match start time.
        # If new time is earlier, adjust Series first, then Matches; if later, adjust Matches first, then Series.
        if series.scheduled_at && startAt < series.scheduled_at
          series.update!(scheduled_at: startAt)
          series.matches.each {|match| match.update!(scheduled_at: startAt)}
        else
          series.matches.each {|match| match.update!(scheduled_at: startAt)}
          series.update!(scheduled_at: startAt)
        end
      end
    end

    # Episode 2 of Smashgg Devs Randomly Destroy Information:
    # `j_set['games']` _becomes_ empty in future responses if a contender was disqualified,
    # even though match win rate still factors in to standings ordering.
    # Decree from Marian is to forge a clean sweep.
    if became_disqualified
      series.matches.each do |match|
        match.opponents.each(&:delete)
        match.delete
      end
      j_set['games'] = [{
        'entrant1Id' => j_set['entrant1Id'],
        'entrant2Id' => j_set['entrant2Id'],
        'entrant1Score' => [score1, 0].max,
        'entrant2Score' => [score2, 0].max,
        'winnerId' => j_set['winnerId'],
      }] * j_set['bestOf']
    end

    j_set['games'].each do |j_game|
      log_match_prefix = log_prefix.chomp(' ') + "[game #{j_game['id']}] "

      if match = Match.find_by("metadata->'smashgg'->>'gameId' = ?", j_game['id'].to_s)
        # Possible updates: competing opponents, scores, winner.
        update_opponents(match, j_game)
        opp1, opp2 = match.opponents.sort_by {|opp| opp.metadata['smashgg']['entrantIndex']}
        score1, score2 = [1,2].map{|i| j_game["entrant#{i}P1Stocks"].to_i}
        if [opp1.score, opp2.score] != [score1, score2]
          log(log_match_prefix + "updating scores: #{opp1.score} to #{opp2.score} -> #{score1} to #{score2}")
          opp1.update!(score: score1) if opp1.score != score1
          opp2.update!(score: score2) if opp2.score != score2
        end

        unless j_game['winnerId'].blank?
          winner_opp = j_game['winnerId'] == opp1.metadata['smashgg']['entrantId'] ? opp1 : opp2
          if winner_opp.winner_id != winner_opp.id
            log(log_match_prefix + "updating winner: #{@teams[j_game['winnerId']].name}")
            winner_opp.update!(winner: winner_opp)
          end
        end

        next
      end

      # Above `if` has `next` as last statement; if we're here, this is a new Match.
      match = series.matches.build(
        game: @game,
        scheduled_at: series.scheduled_at,
        metadata: {'smashgg' => {
          'gameId'     => j_game['id'],
          'setId'      => j_game['setId'],
        }},
      )
      [1,2].each do |i|
        id = j_game["entrant#{i}Id"]
        score = j_game["entrant#{i}P1Stocks"].to_i
        opp = match.opponents.build(
          contender: @teams[id],
          score: score,
          metadata: {'smashgg' => match.metadata['smashgg'].merge({
            'entrantIndex' => i,
            'entrantId'    => id,
          })},
        )
        opp.winner = opp if j_game['winnerId'] == id
      end
      match.save!
      log(log_prefix + "created new Match: #{match.id}")
    end
  end

  def update_opponents(parent, j_parent)
    opponents = parent.opponents.to_a
    opponents_old = Hash[opponents.map {|opp| [opp.contender_id, opp]}]
    opponents_new = Hash[[1,2].map {|i| [@teams.fetch(j_parent["entrant#{i}Id"], nil)&.id, i]}]
    opponents_add = (opponents_new.keys - opponents_old.keys).compact
    opponents_delete = (opponents_old.keys - opponents_new.keys).compact

    return if opponents_add.empty? && opponents_delete.empty?
    log_prefix = "[update_opponents][#{parent.class.name} #{parent.id}] "
    log(log_prefix + "found #{opponents_add.length} new and #{opponents_delete.length} stale opponents")

    opponents_add.each do |team_id|
      entrant_i = opponents_new[team_id]
      team = @teams[j_parent["entrant#{entrant_i}Id"]]
      log(log_prefix + "adding Team #{team.name.inspect} (#{team.id})")

      score_key = "entrant#{entrant_i}" + (parent.is_a?(Series) ? 'Score' : 'P1Stocks')
      score = [j_parent[score_key].to_i, 0].max
      parent.opponents.create!(
        contender_type: 'Team',
        contender_id: team_id,
        score: score,
        metadata: {'smashgg' => parent.metadata['smashgg'].merge({
          'entrantIndex' => entrant_i,
          'entrantId' => j_parent["entrant#{entrant_i}Id"],
        })},
      )
    end

    opponents_delete.each do |team_id|
      opp = opponents_old[team_id]
      log(log_prefix + "removing Team #{opp.contender.name.inspect} (#{opp.contender_id})")
      opp.delete
    end

    parent.reload
  end
end
