require 'rails_helper'
require 'action_dispatch/routing/inspector'

describe Authorization do
  controller(ApplicationController) do
    include Authorization

    require_user! only: [:requires_user]
    require_admin! only: [:requires_admin]
    def requires_nothing; render plain: __method__; end
    def requires_user;    render plain: __method__; end
    def requires_admin;   render plain: __method__; end
    def requires_foo_and_bar
      require_abilities! :foo, :bar, for: 'baz'
      render plain: __method__
    end

    # RSpec does not create new instances of controllers per request within the same test;
    # `reset` is provided to reset internal state between requests.
    def reset
      @user_id = nil
      @current_person = nil
    end

    # `attr_writer`s for easy `double()`ing
    attr_writer :current_person, :user_id
  end

  def json
    JSON.parse(response.body)
  end

  describe 'requires_nothing' do
    it 'should raise no auth/forbidden exceptions if no permissions are required' do
      routes.draw { get 'requires_nothing' => 'anonymous#requires_nothing' }
      get :requires_nothing
      expect(response.status).to eq(200)
    end
  end

  describe 'requires_user' do
    # `requires_user!` does its checks in a `before_action`,
    # so it should render an error reply instead of raising exceptions.
    it 'should reply with HTTP 401 when User-ID header is missing' do
      routes.draw { get 'requires_user' => 'anonymous#requires_user' }
      get :requires_user
      expect(response.status).to eq(401)
      expect(json['error']).to match(/User-ID/)

      controller.reset
      request.headers['User-ID'] = '00000000-0000-0000-0000-000000000000'
      get :requires_user
      expect(response.status).to eq(200)
    end

    it 'should not confuse the user_id param for the User-ID header' do
      routes.draw { get 'requires_user' => 'anonymous#requires_user' }
      get :requires_user,
          params: {'user_id' => '00000000-0000-0000-0000-000000000000'}
      expect(response.status).to eq(401)
      expect(json['error']).to match(/User-ID/)
    end

    it 'should validate that the User-ID header contains a valid v4 UUID' do
      routes.draw { get 'requires_user' => 'anonymous#requires_user' }
      request.headers['User-ID'] = 'not a valid UUID'
      get :requires_user
      expect(response.status).to eq(400)
      expect(json['error']).to match(/user_id.*uuid/i)
    end
  end

  describe 'requires_admin' do
    # Like `requires_user!`, `requires_admin!` also does its checks in a `before_action`,
    # so it should render an error reply instead of raising exceptions.
    it 'should require both a valid User-ID header and X-Roles header' do
      routes.draw { get 'requires_admin' => 'anonymous#requires_admin' }

      get :requires_admin
      expect(response.status).to eq(401)

      controller.reset
      request.headers['X-Roles'] = 'admin'
      get :requires_admin
      expect(response.status).to eq(401)

      controller.reset
      request.headers['X-Roles'] = ''
      request.headers['User-ID'] = '00000000-0000-0000-0000-000000000000'
      get :requires_admin
      expect(response.status).to eq(403)
      expect(json['error']).to match(/admin/i)

      controller.reset
      request.headers['X-Roles'] = 'admin'
      get :requires_admin
      expect(response.status).to eq(200)
    end
  end

  describe 'requires_abilities' do
    # `requires_abilities!` checks occur _within_ an action, so it's expected to raise
    # exceptions for BaseController to handle.
    it 'always requires authentication' do
      routes.draw { get 'requires_foo_and_bar' => 'anonymous#requires_foo_and_bar' }
      expect{get :requires_foo_and_bar}.to raise_error(Authorization::NotAuthenticated)
    end

    it 'should test for abilities before allowing a request' do
      routes.draw { get 'requires_foo_and_bar' => 'anonymous#requires_foo_and_bar' }
      request.headers['User-ID'] = '00000000-0000-0000-0000-000000000000'

      [true, false].repeated_permutation(2).each do |can_foo, can_bar|
        controller.reset
        controller.current_person = double(is_admin?: false, can_foo?: can_foo, can_bar?: can_bar)

        if can_foo && can_bar
          expect{get :requires_foo_and_bar}.to_not raise_error
          expect(response.status).to eq(200)
        else
          expect{get :requires_foo_and_bar}.to raise_error(Authorization::Forbidden)
        end
      end
    end
  end
end
