require 'rails_helper'

describe V1::ArticlesController, type: :controller do
  describe 'filter_params' do
    before :each do
      @controller = V1::ArticlesController.new
      @author = FactoryGirl.create(:person)
      @author_id = @author.user_id
      @other = FactoryGirl.create(:person)
      @other_id = @other.user_id
    end

    def stub_params(params)
      allow(@controller).to receive(:params) { ActionController::Parameters.new(params) }
    end

    def stub_headers(headers)
      allow(@controller).to receive_message_chain(:request, :headers) { headers }
    end

    describe 'for unauthenticated users' do
      before :each do
        stub_headers({})
      end

      it 'should default to published articles when given no filters' do
        stub_params({})
        fparams = @controller.send(:filter_params)
        expect(fparams[:aasm_state]).to eq('final')
        expect(fparams[:published_before]).to be <= Time.now
        expect(fparams[:order]).to eq('published_at DESC')
      end

      it 'should default to published articles when given an author filter' do
        stub_params({author_id: @author_id})
        fparams = @controller.send(:filter_params)
        expect(fparams[:author]).to eq(@author)
        expect(fparams[:aasm_state]).to eq('final')
        expect(fparams[:published_before]).to be <= Time.now
        expect(fparams[:order]).to eq('published_at DESC')
      end

      it 'should reject filters on article state' do
        stub_params({state: 'draft', aasm_state: 'review'})
        expect { @controller.send(:filter_params) }.to raise_error(Authorization::Forbidden)
      end
    end

    describe 'for authors' do
      before :each do
        stub_headers({'User-ID' => @author_id})
      end

      it 'should default to published articles when given no author filter' do
        stub_params({})
        fparams = @controller.send(:filter_params)
        expect(fparams).to_not include(:author_id)
        expect(fparams[:aasm_state]).to eq('final')
        expect(fparams[:published_before]).to be <= Time.now
        expect(fparams[:order]).to eq('published_at DESC')
      end

      it 'should default to published articles when filtering on a different author' do
        stub_params({author_id: @other_id})
        fparams = @controller.send(:filter_params)
        expect(fparams[:author]).to eq(@other)
        expect(fparams[:aasm_state]).to eq('final')
        expect(fparams[:published_before]).to be <= Time.now
        expect(fparams[:order]).to eq('published_at DESC')
      end

      it 'should reject filters on state when viewing articles by other authors' do
        stub_params({author_id: @other_id, state: 'draft'})
        expect { @controller.send(:filter_params) }.to raise_error(Authorization::Forbidden)
      end

      it 'should default to all articles when filtering on themselves as author' do
        stub_params({author_id: @author_id})
        fparams = @controller.send(:filter_params)
        expect(fparams[:author]).to eq(@author)
        expect(fparams).to_not include(:aasm_state)
        expect(fparams).to_not include(:published_before)
        expect(fparams[:order]).to eq('updated_at DESC')
      end

      it 'should rewrite "state" param to "aasm_state" for their own articles' do
        stub_params({author_id: @author_id, state: 'draft'})
        fparams = @controller.send(:filter_params)
        expect(fparams[:author]).to eq(@author)
        expect(fparams).to_not include(:state)
        expect(fparams[:aasm_state]).to eq('draft')
      end
    end

    describe 'for admins' do
      before :each do
        stub_headers({'User-ID' => @author_id, 'X-Roles' => 'admin'})
      end

      it 'should default to all articles' do
        stub_params({})
        fparams = @controller.send(:filter_params)
        expect(fparams.keys & [:aasm_state, :state, :published_before]).to be_empty
        expect(fparams[:order]).to eq('updated_at DESC')
      end
    end
  end
end
