FactoryGirl.define do
  factory :team_invitation do
    association :team
    token { SecureRandom.hex }
    email { Faker::Internet.email }
  end

  factory :tournament_entry do
    association :tournament

    transient do
      team true
    end

    after(:build) do |entry, evaluator|
      entry.entrant ||= evaluator.team ? create(:team, game: entry.tournament.game) : create(:user)
      entry.tournament.tournament_entries << entry
    end
  end

  factory :league do
    sequence(:name) { |n| [Faker::Color.color_name.capitalize, n].join }

    trait :with_seasons do
      transient do
        season_count 2
        tournament_count 0
      end

      after(:create) do |league, evaluator|
        if evaluator.tournament_count > 0
          create_list(:season_with_tournaments, evaluator.season_count,
                      tournament_count: evaluator.tournament_count,
                      league: league)
        else
          create_list(:season, evaluator.season_count, league: league)
        end
      end
    end

    factory :league_with_seasons, traits: [:with_seasons]
    factory :league_with_tournaments, traits: [:with_seasons]
  end

  factory :season do
    sequence(:name) { |n| "Season #{n}" }
    association :league

    trait :with_tournaments do
      transient do
        tournament_count 2
      end

      after(:create) do |season, evaluator|
        create_list(:tournament, evaluator.tournament_count, season: season)
      end
    end

    factory :season_with_tournaments, traits: [:with_tournaments]
  end

  factory :tournament do
    association :game
    association :season
    sequence(:name) { |n| "SEA Qualifier #{n}" }
    entrants_type 'Team'
    aasm_state :registration_open

    after(:create) do |tournament, evaluator|
      tournament.stages << evaluator.stages
      if tournament.stages.empty?
        create(:tournament_stage,
              tournament: tournament,
              name: 'Open Registration',
              starts_at: 7.days.ago,
              ends_at: 3.days.from_now)
        create(:round_robin_stage,
              tournament: tournament,
              name: 'Group Stage',
              starts_at: 4.days.from_now,
              ends_at: 14.days.from_now)
        create(:tournament_stage,
              tournament: tournament,
              name: 'Live Finals',
              starts_at: 15.days.from_now,
              ends_at: 17.days.from_now)
      end
    end
  end

  factory :tournament_stage do
    sequence(:name) { |n| "Stage #{n}" }
    starts_at { 2.days.ago }
    ends_at { 8.days.from_now }
    color { Faker::Color.hex_color }

    transient do
      # If created by association with a child, child may pass up a game to use for tournament.
      game nil
    end

    after(:build) do |stage, evaluator|
      if evaluator.tournament
        evaluator.tournament.stages << stage
      else
        tournament_params = {stages: [stage]}
        tournament_params[:game] = evaluator.game if evaluator.game
        stage.tournament = build(:tournament, **tournament_params)
      end
    end

    factory :round_robin_stage, class: 'TournamentStage::RoundRobin'
    factory :double_elim_stage, class: 'TournamentStage::DoubleElim'
  end

  factory :organization do
    name { Faker::Company.name }
    email_domains []

    trait :with_people do
      transient do
        people_count 5
      end

      after(:create) do |org, evaluator|
        create_list(:person, evaluator.people_count) do |person|
          person.organizations << org
        end
      end
    end

    trait :with_teams do
      transient do
        team_count 5
      end

      after(:create) do |org, evaluator|
        create_list(:team, evaluator.team_count, organization: org)
      end
    end

    factory :organization_with_people, traits: [:with_people]
    factory :organization_with_teams, traits: [:with_teams]
  end

  factory :team do
    sequence(:name) { |n| "#{Faker::Company.buzzword} #{n}" }
    association :organization
    association :game
    country { Faker::Address.country_code }
    settings Hash.new

    trait :with_people do
      transient do
        person_count 3
      end

      after(:create) do |team, evaluator|
        team.add_member!(create(:person, organizations: [team.organization]), ['owner','captain'])
        create_list(:person, evaluator.person_count-1, organizations: [team.organization]).each do |person|
          team.add_member!(person)
        end
      end
    end

    trait :hidden do
      after(:build) do |team, evaluator|
        team.hidden = true
      end
    end

    factory :team_with_people, traits: [:with_people]
  end

  factory :person do
    first_name { Faker::Name.first_name }
    user_id { SecureRandom.uuid }
    organizations []

    trait :with_teams do
      transient do
        team_count 2
      end

      after(:create) do |person, evaluator|
        create_list(:team, evaluator.team_count) do |team|
          team.add_member!(person)
        end
      end
    end

    trait :with_organizations do
      transient do
        organization_count 1
      end

      after(:create) do |person, evaluator|
        create_list(:organization, evaluator.organization_count) do |organization|
          person.organizations << organization
        end
      end
    end

    factory :person_with_teams, traits: [:with_teams]
    factory :person_with_organizations, traits: [:with_organizations]
  end

  factory :game do
    name { Faker::Name.first_name }
  end

  factory :series do
    scheduled_at { 1.day.ago }

    transient do
      matches []
      matches_count 1

      # `:contenders` is used to make Opponents associated with the Series AND its Matches.
      # Either can be overridden with `:series_contenders` and `:match_contenders`.
      # `:contender_type` is inferred from any passed contenders, defaulting to `:team`.
      contender_type nil
      contenders []
      series_contenders nil
      match_contenders nil
    end

    after(:build) do |series, evaluator|
      if evaluator.tournament_stage.blank?
        stage_params = {}
        stage_params[:game] = evaluator.game if evaluator.game
        series.tournament_stage = FactoryGirl.create(:tournament_stage, **stage_params)
      end
      series.game = series.tournament_stage.tournament.game

      if !(contender_type = evaluator.contender_type)
        all = evaluator.contenders + evaluator.series_contenders.to_a + evaluator.match_contenders.to_a
        contender_type = all&.at(0)&.model_name&.to_s&.downcase&.to_sym || :team
      end

      contenders = evaluator.contenders
      contenders_needed = [0, 2 - contenders.length - [evaluator.series_contenders&.length || 0, evaluator.match_contenders&.length || 0].min].max
      contender_params = {}
      contender_params[:game] = series.game if contender_type == :team
      contenders += build_list(contender_type, contenders_needed, contender_params)

      series_contenders = evaluator.series_contenders || []
      series_contenders += contenders if series_contenders.length < 2
      series.opponents = series_contenders.map {|contender| build(:opponent, contest: series, contender: contender)}

      series.matches = evaluator.matches
      series.matches.each do |match|
        if match.scheduled_at && (series.scheduled_at.nil? || match.scheduled_at < series.scheduled_at)
          series.scheduled_at = match.scheduled_at
        end
      end
      matches_needed = [0, evaluator.matches_count - series.matches.length].max
      match_contenders = evaluator.match_contenders || []
      match_contenders = (match_contenders + contenders)[0..1] if match_contenders.length < 2
      series.matches += build_list(:match, matches_needed, series: series, contenders: match_contenders, scheduled_at: series.scheduled_at)
    end

    factory :bo3_series, class: 'Series::Bo3'
    factory :bo5_series, class: 'Series::Bo5'
    factory :bo7_series, class: 'Series::Bo7'
  end

  factory :match do
    state ''

    transient do
      contenders []
      opponents []
      opponents_count 2
    end

    after(:build) do |match, evaluator|
      opponents = evaluator.opponents
      opponents += evaluator.contenders.map {|contender| build(:opponent, contest: match, contender: contender)}
      opponents_needed = [0, evaluator.opponents_count - opponents.length].max
      opponents += build_list(:opponent, opponents_needed, contest: match)
      match.opponents = opponents

      if evaluator.series
        evaluator.series.matches << match
      else
        match.series = build(:series, matches: [match])
      end
      match.game = evaluator.game || match.series.game
    end
  end

  factory :opponent do
    after(:build) do |opponent, evaluator|
      if evaluator.contest
        opponent.contender = evaluator.contender || create(:team, game: evaluator.contest.game)
        opponent.contest.opponents << opponent
      else
        opponent.contest = build(:match, opponents: [opponent])
        opponent.contender = evaluator.contender || create(:team, game: opponent.contest.game)
      end
    end
  end

  factory :curse_channel, class: Hash do
    access_roles                 []
    allow_temporary_child_groups false
    display_category             nil
    display_category_id          '00000000-0000-0000-0000-000000000000'
    force_push_to_talk           false
    group_id                     { SecureRandom.uuid }
    group_title                  { Faker::Team.name }
    hide_call_members_no_access  false
    hide_no_access               false
    is_public                    true
    message_of_the_day           { Faker::Hacker.say_something_smart }
    root_group_id                { SecureRandom.uuid }
    url_path                     { Faker::Internet.url('example.com') }

    skip_create
    initialize_with { Hash[attributes.to_a.map do |attr, val|
      [attr.to_s.camelize.gsub(/Id$/, 'ID'), val]
    end] }
  end

  factory :article do
    association :author, factory: :person
    title { Faker::Hipster.sentence(1) }
    body {{blocks: [
      # very basic sample output of Draft.js' `convertToRaw()`
      {text: Faker::Hipster.paragraph,
       data: {}, depth: 0, entityRanges: [], inlineStyleRanges: [], key: 'foo', type: 'unstyled'}
    ]}}
    aasm_state :draft

    trait :published do
      aasm_state :final
      published_at { Time.now - 30 }
    end
  end

  factory :image do
    image {
      stream = StringIO.new(ImageHelpers.gen_image_data(random: true))
      def stream.original_filename; "doesn't matter"; end
      stream
    }
  end

  factory :event do
    title { Faker::Hipster.sentence(1) }
    slug nil
    description { {} }
    starts_at { 1.day.from_now }
  end
end
