require 'rails_helper'

RSpec.describe Article, type: :model do
  def gen_changes
    FactoryGirl.build(:article).attributes.symbolize_keys.slice(:title, :body, :slug, :league_id, :game_id, :published_at)
  end

  describe 'author association' do
    it "does not delete author when an article is deleted" do
      article = FactoryGirl.create(:article)
      article.destroy!
      expect(article.author.persisted?)
    end
  end

  describe 'parent association' do
    before :each do
      @author = FactoryGirl.create(:person)
      @parent = FactoryGirl.create(:game)
      @article_without_parent = FactoryGirl.create(:article, author: @author)
      @article_with_parent = FactoryGirl.create(:article, author: @author, parent: @parent)
    end

    it 'can any model type' do
      [:game, :league, :series].each do |parent_type|
        parent_name = parent_type.to_s.camelize
        parent_class = parent_name.constantize
        parent = FactoryGirl.create(parent_type)
        article = FactoryGirl.create(:article, parent: parent)
        expect(article).to be_valid
        expect(article.parent).to_not be_nil
        expect(article.parent).to be_a(parent_class)
        expect(article.parent_id).to eq(parent.id)
        expect(article.parent_type).to eq(parent_name)
      end
    end

    it 'does not delete parent when an article is deleted' do
      @article_with_parent.destroy!
      expect(@parent.persisted?)
    end

    it 'does not delete article when parent is deleted' do
      @parent.destroy!
      expect(@article_with_parent.persisted?)
    end

    it 'cannot be created without both id and type' do
      expect{ @article_without_parent.do_update!(parent_id: @parent.id) }.to raise_error(ActiveRecord::RecordInvalid)
      expect(@article_without_parent.errors[:parent][0]).to match(/requires both/)
      @article_without_parent.reload
      @article_without_parent.errors.clear
      expect{ @article_without_parent.do_update!(parent_type: 'Game') }.to raise_error(ActiveRecord::RecordInvalid)
      expect(@article_without_parent.errors[:parent][0]).to match(/requires both/)
    end

    it 'can be updated with a valid type and id' do
      new_parent = FactoryGirl.create(:game)

      @article_without_parent.update!(parent_id: new_parent.id, parent_type: 'Game')
      expect(@article_without_parent.errors).to be_empty
      expect(@article_without_parent.parent).to eq(new_parent)

      @article_with_parent.update!(parent_id: new_parent.id, parent_type: 'Game')
      expect(@article_with_parent.errors).to be_empty
      expect(@article_with_parent.parent).to eq(new_parent)
    end

    [ ['both id and type', {parent_id: nil, parent_type: nil}],
      ['just the id',      {parent_id: nil}],
      ['just the type',    {parent_type: nil}],
    ].each do |description, update_args|
      it "can be removed by setting #{description} to nil" do
        @article_with_parent.update!(**update_args)
        expect(@article_with_parent.errors).to be_empty
        expect(@article_with_parent.parent).to be_nil

        @article_without_parent.update!(**update_args)
        expect(@article_without_parent.errors).to be_empty
        expect(@article_without_parent.parent).to be_nil
      end
    end

    it 'cannot be updated to an unrecognized type' do
      expect{ @article_with_parent.do_update!(parent_type: 'the goddamn batman') }.to raise_error(ActiveRecord::RecordInvalid)
      expect(@article_with_parent.errors[:parent][0]).to match(/invalid.+type/)
      expect{ @article_without_parent.do_update!(parent_id: 123, parent_type: 'the goddamn batman') }.to raise_error(ActiveRecord::RecordInvalid)
      expect(@article_without_parent.errors[:parent][0]).to match(/invalid.+type/)
    end

    it 'cannot be updated to a missing record' do
      expect{ @article_with_parent.do_update!(parent_id: '123') }.to raise_error(ActiveRecord::RecordInvalid)
      expect(@article_with_parent.errors[:parent][0]).to match(/does not exist/)
      expect{ @article_without_parent.do_update!(parent_id: '123', parent_type: 'Game') }.to raise_error(ActiveRecord::RecordInvalid)
      expect(@article_without_parent.errors[:parent][0]).to match(/does not exist/)
    end
  end

  describe 'state' do
    it 'is initially set to "draft"' do
      article = FactoryGirl.build(:article)
      expect(article.state).to eq('draft')
    end

    # Transition tests:
    # There are three relevant params when considering updates:
    #   title, body, and explicitly requested state (if any)
    # Here's every possible combination of updates to relevant params:
    #   (3 start states * 4 request states * 2 title change (true/false) * 2 body change (true/false)
    #     = 48 combinations
    states = [ nil, 'draft', 'review', 'final' ]
    cases = {}
    states.compact.each do |start_state|
      states.each do |requested_state|
        [true, false].repeated_permutation(2).each do |change_title, change_body|
          cases[ [start_state, requested_state, change_title, change_body].freeze ] =
            case start_state
            when 'draft'
              # "draft": "review" or "final" on request, stays "draft" otherwise
              requested_state || 'draft'
            when 'review'
              # "review":
              #   -> "draft" on request, or given title/body change and no state change request
              #   -> "review" if no target state given and no content changes,
              #               or if target state is "review" regardless of content changes
              #   -> "final" if requested and no title/body change
              case [requested_state, change_title || change_body]
                when *[[nil, true], ['draft', true], ['draft', false]] then 'draft'
                when *[[nil, false], ['review', true], ['review', false]] then 'review'
                when ['final', false] then 'final'
              end
            when 'final'
              # "final": can only remain "final"
              'final' unless ![nil,'final'].include?(requested_state)
            end
        end
      end
    end
    cases.each do |params, expected|
      start, request, title, body = params
      example_name =
        "should #{'NOT ' unless expected}#{start == expected ? 'remain' : 'transition from'} " +
        "#{start.inspect} #{"-> #{expected.inspect} " if expected && expected != start}" +
        "#{"when #{request.inspect} is requested and " if request}" +
        "given update to #{case [title,body]
                            when [true,true] then 'both title and body'
                            when [false,false] then 'neither title nor body'
                            else title ? 'title' : 'body'
                            end}"
      it example_name do
        article = FactoryGirl.create(:article, aasm_state: start.to_sym)
        changes = gen_changes
        changes[:state] = request if request
        changes.delete(:title) unless title
        changes.delete(:body) unless body
        if expected.nil?
          expect { article.do_update(changes) }.to raise_error(AASM::InvalidTransition)
        else
          expect(article.do_update(changes)).to eq(true)
        end
        expect(article.state).to eq(expected || start)
      end
    end
  end

  describe 'slug' do
    it 'creates a slug when an article becomes finalized' do
      article = FactoryGirl.create(:article)
      expect(article.state).to eq('draft')
      expect(article.slug).to be_nil

      article.do_update!(state: 'review')
      expect(article.state).to eq('review')
      expect(article.slug).to be_nil

      article.do_update!(state: 'final')
      expect(article.state).to eq('final')
      expect(article.slug).to_not be_blank
    end

    it 'is not overwritten if one was already set' do
      article = FactoryGirl.create(:article, slug: 'existing-slug')
      article.do_update!(state: 'final')
      expect(article.state).to eq('final')
      expect(article.slug).to eq('existing-slug')
    end
  end

  describe 'flags' do
    it 'does not store duplicates' do
      article = FactoryGirl.create(:article, flags: %w{foo bar foo baz foo bar baz foo})
      expect(article.flags.sort).to eq(%w{foo bar baz}.sort)
      article.update!(flags: %w{foo foo foo foo foo})
      expect(article.flags).to eq(%w{foo})
    end
  end
end
