require 'rails_helper'

RSpec.describe Person, :type => :model do
  describe 'handle_user_change' do
    before :each do
      @person = FactoryGirl.create(:person, {
        user_id: SecureRandom.uuid
      })
      @email_domain = Faker::Internet.domain_name.to_s
      @email = "#{Faker::Internet.user_name}@#{@email_domain}"
      @event = Hashie::Mash.new({
        type: 'USER_CHANGE',
        data: {
          id: @person.user_id,
          email: @email,
          diff: { email: [nil, @email] }
        }
      })
    end

    it 'should create a new organization if none is found by Organization#for_email' do
      expect(@person.organizations).to be_empty
      expect(Organization.for_email(@email_domain)).to be_empty
      Person.handle_user_change(@event)
      @person.reload
      expect(@person.organizations).to_not be_empty
      expect(@person.organizations.first.email_domains).to include(@email_domain)
    end

    it 'should remove membership with organizations not affiliated with new email' do
      # User is initially a member of orgs "foo" & "both".
      # After email change, should be a member of "bar" & "both" (and no longer "foo").
      org_foo = FactoryGirl.create(:organization, email_domains: ['foo.example.com'])
      org_bar = FactoryGirl.create(:organization, email_domains: ['bar.example.com'])
      org_both = FactoryGirl.create(:organization, email_domains: ['foo.example.com', 'bar.example.com'])
      @person.update!(organizations: [org_foo, org_both])

      @event.data.email = 'user@bar.example.com'
      @event.data.diff.email[1] = 'user@bar.example.com'
      Person.handle_user_change(@event)

      @person.reload
      expect(@person.organizations.sort).to eq([org_bar, org_both].sort)
    end
  end

  it_behaves_like 'Sociable', for: :person

  it_behaves_like 'a model with images for', :person, :photo
end
