require 'rails_helper'

RSpec.describe Series, :type => :model do
  describe 'opponents' do
    before :each do
      @series = FactoryGirl.create(:series_with_matches, match_count: 3)
    end
  end

  describe 'Bo5' do
    describe 'winner/id' do
      before :each do
        @series = FactoryGirl.create(:bo5_series, matches_count: 5)
      end

      it 'return nil until a contender has won 3 matches' do
        @series.matches[0..3].each_with_index do |match, i|
          match.winner = match.opponents.sort_by(&:contender_id)[i%2]
          expect(@series.winner_id).to be_nil
          expect(@series.winner).to be_nil
        end
      end

      it 'does not count contested wins as wins' do
        @series.matches.each do |match|
          match.winner = match.opponents.sort_by(&:contender_id)[0]
          match.state = 'contested'
        end
        expect(@series.winner_id).to be_nil
        expect(@series.winner).to be_nil
      end

      it 'returns the contender who won' do
        @series.matches.each do |match|
          match.winner = match.opponents.sort_by(&:contender_id)[0]
        end
        winner = @series.matches[0].winner.contender
        expect(@series.winner_id).to eq(winner.id)
        expect(@series.winner).to eq(winner)
      end
    end
  end

  context 'scheduled_at' do
    before :each do
      @matches = FactoryGirl.create_list(:match, 4)
      [1973, 1971, nil, 1972].each_with_index do |time, i|
        @matches[i].update!(scheduled_at: Time.gm(time)) unless time.nil?
      end
      @series = FactoryGirl.build(:series)
      @series.matches = @matches
    end

    it 'defaults to earliest match time, if any' do
      @series.scheduled_at = nil
      @series.save!
      expect(@series.scheduled_at).to eq(Time.gm(1971))
    end

    it 'cannot be before the earliest match time, if any' do
      @series.scheduled_at = Time.now
      @series.save
      expect(@series.errors.keys).to include(:scheduled_at)
      expect(@series.errors[:scheduled_at]).to include('must be before first scheduled match')
    end
  end
end
