require 'rails_helper'

RSpec.describe TeamInvitation, type: :model do
  # Helper to make tests a bit more readable:
  def invite(fields={})
    FactoryGirl.build(:team_invitation, fields)
  end

  it 'requires a team' do
    expect(invite(team: nil)).to_not be_valid
    expect(invite).to be_valid
  end

  describe '#token' do
    it 'defaults to a random token' do
      expect(invite.token).to_not be_empty
    end
  end

  describe '#email' do
    it 'must not be missing or blank' do
      expect(invite(email: nil)).to be_invalid
      expect(invite(email: '')).to be_invalid
    end

    it 'must be a valid email' do
      expect(invite(email: 'not @ valid.email')).to be_invalid
      expect(invite).to be_valid
    end

    it 'validates email uniqueness only' do
      first = invite
      expect(first).to be_valid
      first.save!
      expect(first).to be_persisted
      expect(invite(email: first.email, team: first.team)).to be_invalid
    end

    it 'validates email uniqueness only for unused invitations' do
      first = invite(used: true)
      expect(first).to be_valid
      first.save!
      expect(first).to be_persisted

      second = invite(email: first.email)
      expect(second).to be_valid
      second.save!
      expect(second).to be_persisted
    end
  end

  describe '#accept!' do
    it 'should not work on a used invitation' do
      invitation = invite
      person = FactoryGirl.create(:person)
      expect(invitation.accept(person)).to be(true)
      invitation.reload
      expect(invitation.used).to be(true)
      expect(invitation.accept(person)).to be(false)
    end

    it 'should fail if the user is already on the team' do
      invitation = invite(team: FactoryGirl.create(:team_with_people))
      invitation.save!
      person = invitation.team.people.first
      expect(person).to be_valid
      expect(invitation.accept(person)).to be(false)
      invitation.reload
      expect(invitation.used).to be(false)
    end
  end
end
