require 'rails_helper'

RSpec.describe TeamMember, :type => :model do
  it "should delete all of a user's team memberships when user is deleted" do
    team = FactoryGirl.create(:team_with_people)
    member = team.members.first
    person = member.person
    person.destroy!
    team.reload
    expect(team).to be_persisted
    expect(team.people).to_not include(person)
    expect(team.members).to_not include(member)
  end

  it "should delete all of a team's memberships when team is deleted" do
    team = FactoryGirl.create(:team_with_people)
    people = team.people
    team.destroy!
    expect(TeamMember.where(team_id: team.id).count).to eq(0)
    people.each {|person| expect(person).to be_persisted}
  end

  it 'should not affect associated user or team when membership is deleted' do
    team = FactoryGirl.create(:team_with_people, person_count: 3)
    member = team.members.first
    person = member.person
    member.destroy!
    team.reload
    person.reload
    expect(team).to be_persisted
    expect(person).to be_persisted
    expect(team.members).to_not include(member)
    expect(team.members.count).to eq(2)
    expect(team.people).to_not include(person)
    expect(team.people.count).to eq(2)
  end

  it "should show a member as verified when he's in the team's org" do
    team = FactoryGirl.create(:organization_with_teams, team_count: 1).teams.first
    member = team.add_member!(FactoryGirl.create(:person))
    expect(member.verified).to eq(false)
    member.person.update!(organizations: [team.organization])
    expect(member.verified).to eq(true)
  end
end
