require 'rails_helper'

RSpec.describe Team, :type => :model do
  it 'should not allow names with duplicate teams' do
    t = FactoryGirl.create(:team)
    # For some reason, `validate uniqueness: true` in the model
    # makes it raise a RecordInvalid instead of a RecordNotUnique.
    expect{FactoryGirl.create(:team, name: t.name)}.to \
      raise_error(ActiveRecord::RecordInvalid,
                  'Validation failed: Name has already been taken')
  end

  describe 'slug' do
    it 'should be created based on name' do
      team = FactoryGirl.create(:team)
      expect(team.slug).to_not be_blank
    end
  end

  describe '#country' do
    it 'should allow 2-digit country codes' do
      team = FactoryGirl.create(:team, country: 'us')
      expect(team).to be_valid
      expect(team.country).to eq('United States of America')
      expect(team.region).to eq('Americas')
    end

    it 'should allow 3-digit country codes' do
      team = FactoryGirl.create(:team, country: 'usa')
      expect(team).to be_valid
      expect(team.country).to eq('United States of America')
      expect(team.region).to eq('Americas')
    end

    it 'should country names' do
      team = FactoryGirl.create(:team, country: 'united states')
      expect(team).to be_valid
      expect(team.country).to eq('United States of America')
      expect(team.region).to eq('Americas')
    end

    it 'should raise a validation error for unrecognized countries' do
      expect{ FactoryGirl.create(:team, country: 'foo') }.to \
        raise_error(ActiveRecord::RecordInvalid,
                    'Validation failed: Country is invalid: "foo"')
    end
  end

  describe '#people' do
    before :all do
      @team = FactoryGirl.create(:team)
    end

    it 'should validate uniqueness of a Person on the team' do
      person = FactoryGirl.create(:person)
      @team.add_member!(person)
      expect(@team.people).to include(person)
      expect{@team.add_member!(person)}.to raise_error(ActiveRecord::RecordNotUnique)
      expect(@team.people.count).to eq(1)
    end
  end

  it_behaves_like 'a model with images for', :team, [:logo, :cover]

  context 'organizations' do
    before :all do
      @team = FactoryGirl.create(:team_with_people, person_count: 5)
      @orgs = FactoryGirl.create_list(:organization, 10)
      5.times {|i| @team.people[i].organizations = @orgs[(4-i)..(5+i)]}
    end

    it 'should be listed in descending order by membership by #organizations' do
      expect(@team.organizations.length).to eq(@orgs.length)
      # `#organizations` already uses `#memberships`, so this test shouldn't.
      org_popularity = Hash[*@orgs.map do |org|
        [ org, @team.people.to_a.count {|p| p.organizations.include?(org)} ]
      end.flatten]
      @team.organizations.each_cons(2) do |org1, org2|
        expect(org_popularity[org1]).to be >= org_popularity[org2]
      end
    end

    it 'should be grouped with membership counts by #memberships' do
      memberships = @team.memberships
      # The function for getting these is way more unreadable than just the literal array:
      expected = [1,2,3,4,5,5,4,3,2,1]
      @orgs.each_with_index do |org, i|
        expect(memberships[org.id]).to eq(expected[i])
      end
    end
  end

  it_behaves_like 'Sociable', for: :team

  describe 'settings' do
    it 'defaults to an empty Hash' do
      team = Team.create!(name: FactoryGirl.build(:team).name)
      expect(team.settings).to eq({})
    end

    describe '"charity"' do
      before :each do
        @team = FactoryGirl.create(:team)
      end

      it 'defaults to nil' do
        expect(@team.settings).to_not include('charity')
        expect(@team.settings).to_not include(:charity)
        expect(@team.charity).to eq(nil)
        expect(@team).to be_valid
      end

      it 'can be set to an allowed charity' do
        charity = Team.approved_charities.compact.sample
        @team.charity = charity
        expect(@team).to be_valid
        expect(@team.charity).to eq(charity)
        expect(@team.settings['charity']).to eq(charity)
      end

      it 'can be set to nil' do
        @team.charity = nil
        expect(@team).to be_valid
        expect(@team.charity).to be_nil
        expect(@team.settings).to include('charity')
        expect(@team.settings).to_not include(:charity)
        expect(@team.settings['charity']).to be_nil
      end

      it 'cannot be set to an unapproved charity' do
        bad_charity = 'The Bashar al-Assad Chemical Weapons Procurement Fund'
        @team.charity = bad_charity
        expect(@team).to_not be_valid
        expect(@team.errors[:settings]).to include("unapproved charity: #{bad_charity.inspect}")
      end
    end

    describe '"hidden"' do
      before :each do
        @team = FactoryGirl.create(:team)
      end

      it 'is wrapped by `#hidden` and `#hidden=`' do
        expect(@team.settings).to_not include('hidden')
        expect(@team.hidden).to eq(false)

        @team.hidden = true
        expect(@team.settings).to include('hidden')
        expect(@team.hidden).to eq(true)

        @team.hidden = false
        expect(@team.settings).to include('hidden')
        expect(@team.hidden).to eq(false)
      end

      it 'is always boolean' do
        [true, 'true', 't', 1, '1', 'yes', 'YEPPERONI'].each do |value|
          @team.hidden = value
          expect(@team.hidden).to be_a(TrueClass), "#{value.inspect} not converted to true"
          expect(@team.settings['hidden']).to be_a(TrueClass), "#{value.inspect} not converted to true"
        end

        [false, 0, '0', 'false', 'f', 'no', 'null', 'nil', nil].each do |value|
          @team.hidden = value
          expect(@team.hidden).to be_a(FalseClass), "#{value.inspect} not converted to false"
          expect(@team.settings['hidden']).to be_a(FalseClass), "#{value.inspect} not converted to false"
        end
      end
    end
  end
end
