require 'rails_helper'

RSpec.describe TournamentEntry, type: :model do
  describe 'validations' do
    before :each do
      @tournament_entry = FactoryGirl.create(:tournament_entry)
    end

    it 'requires a tournament' do
      expect(@tournament_entry.tournament).to_not be_nil
      expect(@tournament_entry).to be_valid
      @tournament_entry.tournament = nil
      expect(@tournament_entry).to_not be_valid
    end

    it 'requires an entrant' do
      expect(@tournament_entry.entrant).to_not be_nil
      expect(@tournament_entry).to be_valid
      @tournament_entry.entrant = nil
      expect(@tournament_entry).to_not be_valid
    end

    it 'validates entrant_type against tournament#entrants_type' do
      expect(@tournament_entry).to be_valid
      @tournament_entry.tournament.update_attribute(:entrants_type, 'Invalid')
      expect(@tournament_entry).to be_invalid
    end

    it 'does not invalidate a persisted entry when tournament registration closes' do
      tournament = FactoryGirl.create(:tournament)
      team = FactoryGirl.create(:team)
      entry = tournament.tournament_entries.create(entrant: team)
      expect(entry).to be_valid
      tournament.reset!
      expect(entry).to be_valid
    end

    it 'requires that the tournament state is registration_open on destruction' do
      tournament = FactoryGirl.create(:tournament)
      team = FactoryGirl.create(:team)
      # tournament.open_registration!
      entry = tournament.tournament_entries.create!(entrant: team)
      expect(entry).to be_valid
      tournament.update_attribute(:aasm_state, :registration_closed)
      expect(entry).to be_valid
      expect{entry.destroy}.to raise_error(ActiveRecord::RecordInvalid)
      tournament.open_registration!
      expect{entry.destroy}.to_not raise_error
    end

    it 'requires that the tournament state is registration_open on creation' do
      tournament = FactoryGirl.create(:tournament)
      team = FactoryGirl.create(:team)
      tournament.reset!
      entry = tournament.tournament_entries.create(entrant: team)
      expect(entry).to be_invalid
      tournament.open_registration!
      expect(entry).to be_valid
    end
  end
end
