require 'rails_helper'

RSpec.describe Tournament, type: :model do
  describe 'validations' do
    before :all do
      @tournament = FactoryGirl.create(:tournament)
    end

    it 'requires entrants_type' do
      tournament = Tournament.new(name: 'LEL')
      expect(tournament).to be_invalid
      tournament.save
      expect(tournament.errors).to include(:entrants_type)
    end

    it 'only allows entrants of entrants_type' do
      person = FactoryGirl.create(:person)
      @tournament.update_attribute(:entrants_type, 'Team')
      entry = @tournament.tournament_entries.create({
        entrant_type: 'Person',
        entrant_id: person.id
      })
      expect(entry).to be_invalid
      expect(entry.errors).to include(:entrant_type)
    end
  end

  describe 'starts_at+ends_at' do
    before :all do
      @tournament = FactoryGirl.create(:tournament)
    end

    it 'returns the earliest stage start date in starts_at' do
      expect(@tournament.starts_at).to eq(@tournament.stages.collect(&:starts_at).min)
    end
    it 'returns the latest stage end date in ends_at' do
      expect(@tournament.ends_at).to eq(@tournament.stages.collect(&:ends_at).max)
    end
  end

  describe 'slug' do
    it 'should be created based on league, season, and tournament name' do
      league = FactoryGirl.create(:league, name: 'Foo')
      season = FactoryGirl.create(:season, league: league, name: 'Bar')
      tournament = FactoryGirl.create(:tournament, season: season, name: 'Baz')
      expect(tournament.slug).to eq('foo-bar-baz')
    end
  end

  describe 'stages' do
    before :each do
      @tournament = FactoryGirl.create(:tournament)
    end

    it 'accepts nested attributes for stages' do
      stage = @tournament.stages.first
      attrs = {
        stages_attributes: [{
          id: stage.id, name: 'new stage name'
        }]
      }
      @tournament.update_attributes(attrs)
      @tournament.reload
      expect(@tournament.stages.first.name).to eq('new stage name')

      new_tournament = FactoryGirl.create(:tournament, stages_attributes: [{
        name: 'yet another stage',
        color: 'rgba(0,0,0,0)',
        starts_at: 3.weeks.ago,
        ends_at: 3.days.from_now
      }])
      expect(new_tournament.stages.first.name).to eq('yet another stage')
    end
  end

  it_behaves_like 'a model with images for', :tournament, [:logo, :cover]
end
