require 'rails_helper'

RSpec.describe TournamentStage, type: :model do
  it 'validates color' do
    stage = TournamentStage.new(color: 'wat')
    expect(stage).to be_invalid
    expect(stage.errors).to include(:color)
    stage = TournamentStage.new(color: '#000000')
    expect(stage).to be_invalid # for all the wrong reasons
    expect(stage.errors).to_not include(:color)
  end

  it 'orders by starts_at by default' do
    tournament = FactoryGirl.create(:tournament)
    FactoryGirl.create(:tournament_stage, tournament: tournament,
                       starts_at: tournament.stages.first.starts_at - 3.weeks)
    tournament.reload
    last_stage = tournament.stages.first
    for stage in tournament.stages do
      expect(stage.starts_at).to be >= last_stage.starts_at
      last_stage = stage
    end
  end

  describe '#progress' do
    it 'did the math' do
      stage = FactoryGirl.create(:tournament_stage,
                                 starts_at: 2.days.from_now,
                                 ends_at: 3.days.from_now)
      expect(stage.progress).to eq(0.0)

      stage = FactoryGirl.create(:tournament_stage,
                                 starts_at: 2.days.ago,
                                 ends_at: 2.days.from_now)
      expect(stage.progress.round(3)).to eq(0.5)

      stage = FactoryGirl.create(:tournament_stage,
                                 starts_at: 1.days.ago,
                                 ends_at: 2.days.from_now)
      expect(stage.progress.round(3)).to eq(0.333)

      stage = FactoryGirl.create(:tournament_stage,
                                 starts_at: 2.days.ago,
                                 ends_at: 1.days.ago)
      expect(stage.progress).to eq(1.0)
    end
  end

  describe 'RoundRobin' do
    describe '#standings' do
      # Helper to make a Series for a given stage.
      # score :: { Team => match wins }
      def make_series(stage, score, type=:bo3_series)
        series = FactoryGirl.create(
          type,
          tournament_stage: stage,
          contenders: score.keys,
          matches_count: score.values.inject(0){|a,b|a+b},
        )
        score.sort_by{|team, wins| wins}.inject(0) do |i, pair|
          team, wins = pair
          wins.times do |j|
            match = series.matches[i+j]
            match.update!(winner: match.opponents.detect {|opp| opp.contender == team})
          end
          i + wins
        end
        series
      end

      # Helper to create a standings entry.
      def standing(team:, win:, lose:, mwin:, mlose:)
        {contender: {schema: 'team', id: team.id, name: team.name},
         wins: win, losses: lose, match_wins: mwin, match_losses: mlose}
      end

      it 'should return an empty array for an orphaned stage' do
        # i.e. Don't 500 just because we have an orphaned stage floating around.
        stage = FactoryGirl.create(:round_robin_stage)
        stage.tournament.destroy!
        stage.reload
        expect(stage.tournament).to be_nil
        expect(stage.standings).to be_a(Array)
        expect(stage.standings).to be_empty
      end

      it 'should return an empty array for a tournament without entrants' do
        # Standings need to include 0-0 for entrants that haven't played yet,
        # so if there are no `TournamentEntry`s, we can't give an accurate result.

        stage = FactoryGirl.create(:round_robin_stage)
        teams = FactoryGirl.create_list(:team, 5, game: stage.tournament.game)
        # Do NOT add these teams as tournament entrants as we normally would:
        # stage.tournament.teams += teams

        # Create an otherwise valid set of series and matches.
        teams.combination(2).each do |pair|
          scores = Hash[ pair.zip([2,0]) ] # 2-0 in a Bo3
          make_series(stage, scores)
        end
        stage.reload
        expect(stage.series).to_not be_empty
        stage.series.each {|series| expect(series.winner).to_not be_nil}

        # Standings should be empty nonetheless.
        expect(stage.standings).to be_a(Array)
        expect(stage.standings).to be_empty
      end

      it 'should exclude hidden teams' do
        stage = FactoryGirl.create(:round_robin_stage)
        stage.tournament.teams += teams = FactoryGirl.create_list(:team, 5, game: stage.tournament.game)
        teams[2].hidden = true; teams[2].save!

        # Give hidden team some Series and Matches as well to make sure `#standings` doesn't choke on them.
        scores = [ {teams[0] => 2, teams[2] => 0}, {teams[1] => 2, teams[2] => 0} ]
        scores.each {|score| make_series(stage, score)}

        result_teams = stage.standings.map {|s| s[:team]}
        expect(result_teams.length).to eq(4)
        expect(result_teams).to_not include(teams[2])
      end

      it 'returns an array sorted by wins' do
        # Test data comes from the LCS NA Challenger 2017 Summer Split.
        lol = FactoryGirl.create(:game, name: 'League of Legends')
        stage = FactoryGirl.create(:round_robin_stage, game: lol)

        teams = [
          eun = FactoryGirl.create(:team, game: lol, name: 'eUnited'),
          gcu = FactoryGirl.create(:team, game: lol, name: 'Gold Coin United'),
          ts  = FactoryGirl.create(:team, game: lol, name: 'Tempo Storm'),
          big = FactoryGirl.create(:team, game: lol, name: 'Big Gods Jackals'),
          cla = FactoryGirl.create(:team, game: lol, name: 'CLG Academy'),
          dfx = FactoryGirl.create(:team, game: lol, name: 'Delta Fox'),
        ]
        stage.tournament.teams += teams

        [ {gcu => 0, ts  => 2}, {big => 2, cla => 0}, {eun => 2, dfx => 0}, # Week 1, Day 1
          {big => 0, ts  => 2}, {dfx => 0, cla => 2}, {eun => 2, gcu => 1}, # Week 1, Day 2
          {gcu => 2, big => 0}, {ts  => 2, dfx => 0}, {cla => 0, eun => 2}, # Week 2, Day 1
          {cla => 1, gcu => 2}, {eun => 2, ts  => 1}, {dfx => 0, big => 2}, # Week 2, Day 2
          {big => 2, dfx => 0}, {ts  => 1, eun => 2}, {gcu => 2, cla => 0}, # Week 3, Day 1
          {dfx => 1, eun => 2}, {big => 1, gcu => 2}, {cla => 1, ts  => 2}, # Week 3, Day 2
          {gcu => 2, dfx => 0}, {ts  => 2, big => 0}, {eun => 2, cla => 0}, # Week 4, Day 1
          {big => 0, eun => 2}, {cla => 2, dfx => 0}, {ts  => 1, gcu => 2}, # Week 4, Day 2
          {gcu => 2, eun => 0}, {dfx => 1, ts  => 2}, {cla => 0, big => 2}, # Week 5, Day 1
          {ts  => 2, cla => 0}, {dfx => 1, gcu => 2}, {eun => 2, big => 0}, # Week 6, Day 2
        ].each do |score|
          series = make_series(stage, score)
          expect(series.winner_id).to_not be_nil
        end

        expected = [
          standing(team: eun, win: 9, lose:  1, mwin: 18, mlose:  6),
          standing(team: gcu, win: 8, lose:  2, mwin: 17, mlose:  8),
          standing(team:  ts, win: 7, lose:  3, mwin: 17, mlose:  8),
          standing(team: big, win: 4, lose:  6, mwin:  9, mlose: 12),
          standing(team: cla, win: 2, lose:  8, mwin:  6, mlose: 16),
          standing(team: dfx, win: 0, lose: 10, mwin:  3, mlose: 20),
        ]
        actual = stage.standings
        expect(actual).to eq(expected)
      end

      it 'sorts standings correctly' do
        stage = FactoryGirl.create(:round_robin_stage)
        tournament = stage.tournament
        tournament.teams += teams = FactoryGirl.create_list(:team, 10).sort_by {|team| team.name}

        expected = [
          standing(team: teams[0], win: 2, lose: 0, mwin: 4, mlose: 0), # highest series win%
          standing(team: teams[1], win: 1, lose: 0, mwin: 2, mlose: 0), # also 100% win rate, but fewer total series wins
          standing(team: teams[2], win: 1, lose: 1, mwin: 3, mlose: 3), # 50% series win rate, 50% match win rate
          standing(team: teams[3], win: 1, lose: 1, mwin: 2, mlose: 3), # 50% series win rate, 40% match win rate
          standing(team: teams[4], win: 0, lose: 0, mwin: 1, mlose: 0), # 0-0, but won 1 match
          standing(team: teams[5], win: 0, lose: 0, mwin: 0, mlose: 0), # never played
          standing(team: teams[6], win: 0, lose: 0, mwin: 0, mlose: 0), # never played, name sorted after teams[8]
          standing(team: teams[7], win: 0, lose: 0, mwin: 0, mlose: 1), # 0-0, but lost 1 match
          standing(team: teams[8], win: 0, lose: 1, mwin: 1, mlose: 2), # >0 losses
          standing(team: teams[9], win: 0, lose: 2, mwin: 0, mlose: 4), # even more losses
        ]

        # These Bo3 Series produce the above standings:
        [ {teams[0] => 2, teams[3] => 0},
          {teams[0] => 2, teams[9] => 0},
          {teams[1] => 2, teams[9] => 0},
          {teams[2] => 2, teams[8] => 1},
          {teams[2] => 1, teams[3] => 2},
          {teams[4] => 1, teams[7] => 0}, # not concluded
        ].each {|score| make_series(stage, score)}

        expect(stage.standings).to eq(expected)
      end

      it 'really sorts standings correctly' do
        # Real-world example for extra coverage:
        stage = FactoryGirl.create(:round_robin_stage)
        tournament = stage.tournament
        tournament.teams += [
          alg = FactoryGirl.create(:team, name: 'Allegiance'),
          c9  = FactoryGirl.create(:team, name: 'Cloud9'),
          fly = FactoryGirl.create(:team, name: 'FlyQuest'),
          g2  = FactoryGirl.create(:team, name: 'G2 Esports'),
          gho = FactoryGirl.create(:team, name: 'Ghost'),
          nrg = FactoryGirl.create(:team, name: 'NRG Esports'),
          ren = FactoryGirl.create(:team, name: 'Renegades'),
          rog = FactoryGirl.create(:team, name: 'Rogue'),
        ]

        [ { nrg => 3, ren => 0 },
          { c9  => 3, fly => 0 },
          { c9  => 3, alg => 1 },
          { nrg => 1, gho => 3 },
          { ren => 1, g2  => 3 },
          { c9  => 3, rog => 1 },
          { nrg => 3, alg => 1 },
          { fly => 2, rog => 3 },
          { alg => 0, rog => 3 },
          { nrg => 3, rog => 1 },
        ].each {|score| make_series(stage, score, :bo5_series)}

        expected = [
          standing(team: c9,  win: 3, lose: 0, mwin:  9, mlose: 2),
          standing(team: nrg, win: 3, lose: 1, mwin: 10, mlose: 5),
          standing(team: rog, win: 2, lose: 2, mwin:  8, mlose: 8),
          standing(team: g2,  win: 1, lose: 0, mwin:  3, mlose: 1),
          standing(team: gho, win: 1, lose: 0, mwin:  3, mlose: 1),
          standing(team: fly, win: 0, lose: 2, mwin:  2, mlose: 6),
          standing(team: ren, win: 0, lose: 2, mwin:  1, mlose: 6),
          standing(team: alg, win: 0, lose: 3, mwin:  2, mlose: 9),
        ]

        expect(stage.standings).to eq(expected)
      end
    end
  end
end
