require 'rails_helper'

describe V1::PersonPresenter do
  describe 'attributes' do
    before :all do
      @person = FactoryGirl.create(:person, :with_organizations, :with_teams)
      @presenter = V1::PersonPresenter.new(@person)
      @hash = @presenter.to_hash
    end

    [ {name: :first_name,       type: String, can_be_nil: false},
      {name: :last_name,        type: String, can_be_nil: false},
      {name: :handle,           type: String, can_be_nil: true},
      {name: :user_id,          type: String, can_be_nil: false},
      {name: :social,           type: Hash,   can_be_nil: false},
      {name: :organization_ids, type: Array,  can_be_nil: false},
      {name: :team_ids,         type: Array,  can_be_nil: false},
      {name: :organizations,    type: Array,  can_be_nil: false},
      {name: :teams,            type: Array,  can_be_nil: false},
      {name: :photo,            type: Hash,   can_be_nil: false},
    ].each do |attr|
      it "includes #{attr[:name]}" do
        expect(@hash).to include(attr[:name])
        value = @hash[attr[:name]]
        expect(value).to_not be_nil unless attr[:can_be_nil]
        expect(value).to be_a(attr[:type]) unless value.nil?
      end
    end
  end

  describe 'teams' do
    it 'should not embed hidden teams' do
      person = FactoryGirl.create(:person, :with_teams, team_count: 5)
      [1,3].each {|i| person.teams[i].hidden = true; person.teams[i].save!}
      person.reload
      visible_ids = person.teams.reject(&:hidden).map(&:id).sort

      json = V1::PersonPresenter.new(person).as_json
      expect(json[:teams].length).to eq(3)
      expect(json[:team_ids].sort).to eq(visible_ids)
      expect(json[:teams].map{|o|o[:id]}.sort).to eq(visible_ids)
      expect(json[:teams].all?{|team| !team[:hidden]})
    end
  end
end
