require 'rails_helper'

describe V1::SeriesPresenter do
  before :all do
    @series = FactoryGirl.create(:series)
    @presenter = V1::SeriesPresenter.new(@series)
  end

  describe 'attributes' do
    it 'should include all specified attributes' do
      attrs = [:id, :game_id, :tournament_stage_id, :type, :scheduled_at,
               :match_ids, :matches, :opponents, :winner_id]
      expect((@presenter.as_json.keys & attrs).sort).to eq(attrs.sort)
    end

    it 'should return opponents if they are set' do
      series = FactoryGirl.create(:series)
      teams = FactoryGirl.create_list(:team, 2)
      team_ids = teams.map(&:id).sort

      presenter = Hashie::Mash.new(V1::SeriesPresenter.new(series).as_json)
      contender_ids = presenter.opponents.map(&:contender_id)
      expect(contender_ids & team_ids).to be_empty

      teams.each {|team| series.opponents.create(contender: team)}
      presenter = Hashie::Mash.new(V1::SeriesPresenter.new(series).as_json)
      contender_ids = presenter.opponents.map(&:contender_id)
      expect((contender_ids & team_ids).sort).to eq(team_ids)
    end

    it 'should always order opponents by their IDs' do
      def sorted_by_id?(objects)
        objects.each_cons(2).all? {|a,b| a.id < b.id}
      end
      # Make sure opponents are unsorted in input series.
      while sorted_by_id?(@series.opponents)
        FactoryGirl.create(:opponent, contest: @series)
      end

      json = Hashie::Mash.new(V1::SeriesPresenter.new(@series).as_json)
      expect(sorted_by_id?(json.opponents)).to eq(true)
      expect(json.opponents.map(&:id)).to eq(@series.opponents.map(&:id).sort)
    end

    it 'should include winner_to, loser_to, round, and ident for elimination stages' do
      stage = FactoryGirl.create(:double_elim_stage)
      metadata = {round: 123, ident: 'an ident', winner_to: 'a winner id', loser_to: 'a loser id'}
      series = FactoryGirl.create(:series, tournament_stage: stage, metadata: metadata)

      json = V1::SeriesPresenter.new(series).as_json
      metadata.each do |attr, value|
        expect(json).to include(attr)
        expect(json[attr]).to eq(value)
      end
    end
  end

  describe 'Bo5' do
    describe 'attributes' do
      it 'should include all Series attributes, plus winner_id' do
        series = FactoryGirl.create(:bo5_series)
        presenter = V1::SeriesPresenter.new(series)
        attrs = [:id, :game_id, :tournament_stage_id, :type, :scheduled_at,
                 :match_ids, :matches, :opponents, :winner_id]
        expect((presenter.as_json.keys & attrs).sort).to eq(attrs.sort)
      end
    end
  end
end
