require 'rails_helper'

describe V1::ArticlesController do
  it_behaves_like 'a CRUD API', {
    for: :article,
    factory_options: { aasm_state: 'final', published_at: Time.now },
  }, ->{{
    create: {title: 'a title', body: 'a body'},
    update: {slug: 'a-slug'},
  }}

  before :all do
    @authors = FactoryGirl.create_list(:person, 2)
    @articles = @authors.product(['draft','review','final']).map do |author, state|
      options = {author_id: author.id, aasm_state: state}
      options[:published_at] = Time.now if state == 'final'
      FactoryGirl.create(:article, options)
    end
    # Add some finalized but with no publish date, and some with future publish date:
    @articles += @authors.map do |author|
      [FactoryGirl.create(:article, author: author, aasm_state: 'final', published_at: nil),
       FactoryGirl.create(:article, author: author, aasm_state: 'final', published_at: Time.now + 24*60*60)]
    end.flatten
  end

  # RSpec scoping is dumb: `@authors` is only defined within individual examples,
  # so `:author0` is a stand-in for `@authors[0].user_id` to be replaced per example.
  clients = [
    {user_id: nil,                 description: 'anonymous users'},
    {user_id: SecureRandom.uuid,   description: 'users'          },
    {user_id: :author0,            description: 'other authors'  }
  ]

  describe '#index' do
    clients.each do |client|
      it "should show only published articles to #{client[:description]} when given no author filters" do
        user_id = client[:user_id] == :author0 ? @authors[0].user_id : client[:user_id]
        headers = user_id.nil? ? {} : {'User-ID' => user_id}
        get '/v1/articles', headers: headers
        expect(response.status).to eq(200), ->{pp json}
        expect(json_ids.sort).to eq(@articles.select(&:is_published?).map(&:id).sort)
      end

      it "should show only a specific author's published articles to #{client[:description]}" do
        user_id = client[:user_id] == :author0 ? @authors[0].user_id : client[:user_id]
        headers = user_id.nil? ? {} : {'User-ID' => user_id}
        get '/v1/articles', headers: headers, params: {author_id: @authors[1].user_id}
        expect(response.status).to eq(200), ->{pp json}
        exp_ids = @articles.select do |article|
          article.is_published? && article.author == @authors[1]
        end.map(&:id).sort
        expect(json_ids.sort).to eq(exp_ids), ->{pp json}
      end
    end

    it 'should show an author all of their own articles' do
      get '/v1/articles',
        headers: {'User-ID' => @authors[0].user_id},
        params: {author_id: @authors[0].user_id}
      expect(response.status).to eq(200), ->{pp json}
      exp_ids = @articles.select{|article| article.author == @authors[0]}.map(&:id).sort
      expect(json_ids.sort).to eq(exp_ids)
    end

    it 'should show only published articles of one author to another' do
      get '/v1/articles',
        headers: {'User-ID' => @authors[0].user_id},
        params: {author_id: @authors[1].user_id}
      expect(response.status).to eq(200), ->{pp json}
      exp_ids = @articles.select do |article|
        article.is_published? && article.author == @authors[1]
      end.map(&:id).sort
      expect(json_ids.sort).to eq(exp_ids)
    end

    it 'should show admins all articles, including unpublished' do
      headers = {'User-ID' => @authors[0].user_id, 'X-Roles' => 'admin'}
      get '/v1/articles', headers: headers
      expect(response.status).to eq(200), ->{pp json}
      expect(@articles.map(&:id) - json_ids).to be_empty
    end

    it 'can filter by parent given both id and type' do
      game1, game2 = FactoryGirl.create_list(:game, 2)
      game1_articles = FactoryGirl.create_list(:article, 5, :published, parent: game1)
      game2_articles = FactoryGirl.create_list(:article, 5, :published, parent: game2)

      get "/v1/articles", params: {parent_id: game1.id}
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/parent filter .+ id and type/)

      get "/v1/articles", params: {parent_type: 'game'}
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/parent filter .+ id and type/)

      get "/v1/articles", params: {parent_id: game1.id, parent_type: 'the goddamn batman'}
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/invalid parent type: .*batman/i)

      get "/v1/articles", params: {parent_id: game1.id, parent_type: 'game'}
      expect(response.status).to eq(200), ->{pp json}
      expect(json_ids.sort).to eq(game1_articles.map(&:id).sort)
    end

    it 'can filter by flags' do
      article_ids = Hash[{
        tsm: ['tsm'],
        tsm_featured: ['tsm', 'featured'],
        clg: ['clg'],
        clg_featured: ['clg', 'featured'],
        featured: ['featured'],
        none: [],
      }.map do |label, flags|
        [label, FactoryGirl.create_list(:article, 10, :published, flags: flags).map(&:id)]
      end]

      get '/v1/articles', params: {flags: 'featured'}
      expect(response.status).to eq(200)
      featured_ids = article_ids.values_at(:featured, :tsm_featured, :clg_featured).flatten.sort
      expect(json_ids.sort).to eq(featured_ids)

      get '/v1/articles', params: {flags: 'tsm'}
      expect(response.status).to eq(200)
      tsm_ids = article_ids.values_at(:tsm, :tsm_featured).flatten.sort
      expect(json_ids.sort).to eq(tsm_ids)

      get '/v1/articles', params: {flags: 'tsm,featured'}
      expect(response.status).to eq(200)
      expect(json_ids.sort).to eq(article_ids[:tsm_featured].sort)

      get '/v1/articles', params: {flags: 'these,flags,do,not,exist'}
      expect(response.status).to eq(200)
      expect(json).to eq([])
    end
  end

  describe '#show' do
    it 'should not show unpublished articles to anyone but their authors' do
      @articles.each do |article|
        [nil, SecureRandom.uuid, @authors[0].user_id].each do |user_id|
          headers = user_id ? {'user-id' => user_id} : {}
          get "/v1/articles/#{article.id}", headers: headers
          if article.is_published? || user_id == article.author.user_id
            expect(response.status).to eq(200), ->{pp response.body}
            expect(json['id']).to eq(article.id)
          else
            expect(response.status).to eq(403), ->{pp response.body}
          end
        end
      end
    end

    it 'should be able to fetch published articles by slug' do
      article = @articles.detect(&:is_published?)
      expect(article).to_not be_nil
      expect(article.slug).to_not be_blank
      get "/v1/articles/#{article.slug}"
      expect(response.status).to eq(200), ->{pp response.body}
      expect(json['id']).to eq(article.id)
      expect(json['slug']).to eq(article.slug)
    end
  end

  describe '#create' do
    before :each do
      @author = FactoryGirl.create(:person)
      @headers = {'User-ID' => @author.user_id}
      @params = {title: 'test title', body: 'test body'}
    end

    it 'should require a User-ID header' do
      post '/v1/articles', params: @params
      expect(response.status).to eq(401)
    end

    it 'should require a title and body' do
      [true,false].repeated_permutation(2).each do |has_title, has_body|
        params = {}
        params[:title] = 'test title' if has_title
        params[:body] = 'test body' if has_body
        post '/v1/articles', headers: @headers, params: params
        expect(response.status).to eq(has_title && has_body ? 200 : 400)
      end
    end

    it "should register the creator as the article's author" do
      post '/v1/articles', headers: @headers, params: @params
      expect(response.status).to eq(200)
      # The reply should expose the author's User ID,
      # but the association stored in the db is with Person.
      expect(json['author_id']).to eq(@author.user_id)
      article = Article.find(json['id'])
      expect(article.author == @author)
    end

    it 'should allow creation with "draft" and "review" state, but not "final"' do
      ['draft', 'review', 'final'].each do |state|
        @params[:state] = state
        post '/v1/articles', headers: @headers, params: @params
        if state != 'final'
          expect(response.status).to eq(200)
          expect(json['state']).to eq(state)
        else
          expect(response.status).to eq(400)
        end
      end
    end

    it 'should accept article covers' do
      image_file = ImageHelpers.gen_image_file(random: true)
      params = begin
        article = FactoryGirl.build(:article, author: nil)
        { title: article.title,
          body: article.body.to_json,
          cover: Rack::Test::UploadedFile.new(ImageHelpers.gen_image_file(random: true)) }
      end
      post '/v1/articles', headers: @headers, params: params
      expect(response.status).to eq(200), ->{ pp json }
      expect(json.dig('cover','original')).to_not be_blank
    end

    it 'should accept any existing object as a parent' do
      game = FactoryGirl.create(:game)
      params = @params.merge(parent_id: game.id, parent_type: 'game')
      post '/v1/articles', headers: @headers, params: params
      expect(response.status).to eq(200), ->{ pp json }
      expect(json['parent_type']).to eq('Game') # server will camelize the name
      expect(json['parent_id']).to eq(game.id)
    end

    it "should reply with HTTP 400 for a parent type that doesn't exist" do
      params = @params.merge(parent_id: 123, parent_type: 'Kappa')
      post '/v1/articles', headers: @headers, params: params
      expect(response.status).to eq(400), ->{ pp json }
      expect(json['error']).to match(/Parent has invalid type.*kappa/i)
    end

    it 'should reply with HTTP 400 for a non-existent ID for an otherwise valid parent type' do
      params = @params.merge(parent_id: 123, parent_type: 'Game')
      post '/v1/articles', headers: @headers, params: params
      expect(response.status).to eq(400), ->{ pp json }
      expect(json['error']).to match(/parent does not exist/i)
    end

    it 'should reply with helpful error messages for duplicate slugs' do
      article = FactoryGirl.create(:article, slug: 'foo-slug')
      params = @params.merge(slug: article.slug)
      post '/v1/articles', headers: @headers, params: params
      expect(response.status).to eq(400), ->{ pp json }
      expect(json['error']).to eq('Record already exists with this slug')
    end
  end

  describe '#update' do
    before :each do
      @author = FactoryGirl.create(:person)
      @headers = {'User-ID' => @author.user_id}
    end

    it 'allows updates to anything in the draft state' do
      article = FactoryGirl.create(:article, author: @author)
      [nil, 'draft', 'review', 'final'].each do |new_state|
        params = {title: 'test title', body: 'test body'}
        params[:state] = new_state if new_state
        patch "/v1/articles/#{article.id}", headers: @headers, params: params
        if new_state == 'final'
          expect(response.status).to eq(403) # admin, lol
        else
          expect(response.status).to eq(200)
          expect(json['id']).to eq(article.id)
          expect(json['title']).to eq('test title')
          expect(json['body']).to eq('test body')
          expect(json['state']).to eq(new_state || 'draft')
        end
      end
    end

    it 'reverts articles in review to drafts if content changes are made' do
      article = FactoryGirl.create(:article, author: @author, aasm_state: 'review')
      patch "/v1/articles/#{article.id}", headers: @headers, params: {title: 'new title'}
      expect(response.status).to eq(200)
      expect(json['id']).to eq(article.id)
      expect(json['state']).to eq('draft')
    end

    it 'only allows admins to approve articles' do
      article = FactoryGirl.create(:article, author: @author, aasm_state: 'review')
      patch "/v1/articles/#{article.id}", headers: @headers, params: {state: 'final'}
      expect(response.status).to eq(403)
      @headers['X-Roles'] = 'admin'
      patch "/v1/articles/#{article.id}", headers: @headers, params: {state: 'final'}
      expect(response.status).to eq(200)
      expect(json['id']).to eq(article.id)
      expect(json['state']).to eq('final')
    end

    it 'replies with HTTP 400 for invalid state transitions' do
      article = FactoryGirl.create(:article, author: @author, aasm_state: 'final')
      patch "/v1/articles/#{article.id}", headers: @headers,
        params: {state: 'draft', title: 'foo'}
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/cannot transition final -> draft with content changes/)
    end
  end

  describe '#destroy' do
    it 'only allows authors to delete their own unpublished articles' do
      article = FactoryGirl.create(:article)
      delete "/v1/articles/#{article.id}"
      expect(response.status).to eq(401)
      delete "/v1/articles/#{article.id}", headers: {'User-ID' => SecureRandom.uuid}
      expect(response.status).to eq(403)
      delete "/v1/articles/#{article.id}", headers: {'User-ID' => article.author.user_id}
      expect(response.status).to eq(200)
    end

    it 'only allows admins to delete published articles' do
      article = FactoryGirl.create(:article, aasm_state: 'final', published_at: Time.now)
      author_uid = article.author.user_id
      delete "/v1/articles/#{article.id}"
      expect(response.status).to eq(401)
      delete "/v1/articles/#{article.id}", headers: {'User-ID' => SecureRandom.uuid}
      expect(response.status).to eq(403)
      delete "/v1/articles/#{article.id}", headers: {'User-ID' => author_uid}
      expect(response.status).to eq(403)
      delete "/v1/articles/#{article.id}", headers: {'User-ID' => author_uid, 'X-Roles' => 'admin'}
      expect(response.status).to eq(200)
    end
  end
end
