require 'rails_helper'

describe V1::ImagesController do
  describe '#index' do
    it 'returns an array of image metadata objects' do
      images = FactoryGirl.create_list(:image, 5)
      get '/v1/images'
      expect(response.status).to eq(200)
      expect(json).to be_a(Array)
      expect(json.length).to be >= images.length
    end

    it 'supports filtering by tag' do
      foo_images = FactoryGirl.create_list(:image, 3, tags: ['foo'])
      bar_images = FactoryGirl.create_list(:image, 3, tags: ['bar'])
      foobar_images = FactoryGirl.create_list(:image, 3, tags: ['foo','bar'])

      foo_ids = (foo_images + foobar_images).map(&:id).sort
      bar_ids = (bar_images + foobar_images).map(&:id).sort

      get '/v1/images', params: {tags: ['foo']}
      expect(response.status).to eq(200)
      expect(json_ids.sort).to eq(foo_ids)

      get '/v1/images', params: {tags: ['bar']}
      expect(response.status).to eq(200)
      expect(json_ids.sort).to eq(bar_ids)
    end
  end

  describe '#show' do
    before :all do
      @image = FactoryGirl.create(:image)
    end

    it 'replies with metadata about the image if image ID does not have an extension' do
      get "/v1/images/#{@image.id}"
      expect(response.status).to eq(200)
      expect(response.headers['Content-Type']).to match(/^application\/json/)
      expect(json).to be_a(Hash)
    end

    it 'replies with an image in the requested format if an extension is provided' do
      ['jpeg', 'png', 'gif'].each do |format|
        get "/v1/images/#{@image.id}.#{format}"
        expect(response.status).to eq(200), ->{pp json}
        expect(response.headers['Content-Type']).to eq("image/#{format}")
        expect(FileMagic.new(:mime_type).buffer(response.body)).to eq("image/#{format}")
      end
    end

    describe 'transformations' do
      it 'resizes images with the "size" param' do
        get "/v1/images/#{@image.id}.png", params: {size: '40x40'}
        expect(response.status).to eq(200), ->{pp json}
        expect(response.headers['Content-Type']).to eq("image/png")
        # MiniMagick is super unfriendly to in-memory images; everything has to be a gorram file. x_x
        Tempfile.create('') do |temp|
          temp.binmode
          temp.write(response.body)
          temp.close
          expect(MiniMagick::Image.new(temp.path).dimensions).to eq([40,40])
        end
      end
    end

    it 'replies with 400 for invalid image types' do
      get "/v1/images/#{@image.id}.doc"
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/invalid image transformation/i)
      File.unlink('doc:-') if File.exists?('doc:-') # MiniMagick is a piece of shit.
    end

    it 'replies with 400 for invalid transformation arguments' do
      get "/v1/images/#{@image.id}.png?size=pretty-big"
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/invalid image transformation/i)
    end

    it 'replies with 400 for unsupported transformations' do
      get "/v1/images/#{@image.id}.png?combobulation=fierce"
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/unsupported transforms: combobulation/i)
    end
  end

  describe '#create' do
    before :each do
      @data = FactoryGirl.build(:image).data
    end

    it 'should accept an image as the POST body' do
      post '/v1/images',
        headers: {'Content-Type' => 'image/png', 'RAW_POST_DATA' => @data}
      expect(response.status).to eq(200)
      expect(json['id']).to eq(Digest::SHA1.hexdigest(@data))
    end

    it 'should accept an image as form data' do
      Tempfile.open('edb-multipart-form-png') do |tmpfile|
        tmpfile.binmode
        tmpfile.write(@data)
        tmpfile.close
        post '/v1/images',
          params: {image: Rack::Test::UploadedFile.new(tmpfile.path, 'image/png')},
          headers: {'Content-Type' => 'multipart/form-data'}
        expect(response.status).to eq(200), ->{ pp json }
        expect(json['id']).to eq(Digest::SHA1.hexdigest(@data))
      end
    end

    it 'should reject any other content type' do
      post '/v1/images',
        headers: {'Content-Type' => 'application/octet-stream', 'RAW_POST_DATA' => @data}
      expect(response.status).to eq(415)
    end

    it 'should return the existing image when a duplicate is uploaded' do
      existing = FactoryGirl.create(:image)
      post '/v1/images',
        headers: {'Content-Type' => 'image/png', 'RAW_POST_DATA' => existing.data}
      expect(response.status).to eq(200)
      expect(json['id']).to eq(existing.id)
    end
  end

  describe '#update' do
    before :each do
      @image = FactoryGirl.create(:image)
    end

    it 'should modify the image metadata' do
      expect(@image.tags).to be_empty
      tags = ['foo','bar','baz']
      patch "/v1/images/#{@image.id}", params: {tags: tags}
      expect(response.status).to eq(200)
      expect(json['id']).to eq(@image.id)
      expect(json['tags']).to eq(tags)
      @image.reload
      expect(@image.tags).to eq(tags)
    end
  end

  describe '#destroy' do
    it 'should only allow an admin to delete images' do
      user_id = FactoryGirl.create(:person).user_id
      image = FactoryGirl.create(:image)

      delete "/v1/images/#{image.id}"
      expect(response.status).to eq(401)
      delete "/v1/images/#{image.id}", headers: {'User-ID' => user_id}
      expect(response.status).to eq(403)
      delete "/v1/images/#{image.id}", headers: {'User-ID' => user_id, 'X-Roles' => 'admin'}
      expect(response.status).to eq(200)
      expect(json['id']).to eq(image.id)
      expect(Image.exists?(image.id)).to eq(false)
    end
  end
end
