require 'rails_helper'

describe V1::MatchesController do
  it_behaves_like 'a CRUD API', { for: :match }, -> {{
    create: FactoryGirl.build(:match).attributes.merge({
      opponents: 2.times.collect {{
        contender_id: FactoryGirl.create(:team).id,
        contender_type: 'Team'
      }}
    }),
    update: { game_id: FactoryGirl.create(:game).id }
  }}

  describe '#index' do
    it 'finds matches a user is involved in when given a user_id' do
      # Helper to make creating lots of test Matches less annoying:
      def new_match(game, *contenders)
        Match.create!(game: game, opponents: contenders.map do |c|
          c = FactoryGirl.create(:person) if c == :rando
          Opponent.create(contender: c)
        end).id
      end

      # Don't need a Game to make a Match, but we use one here
      # to check that other filters are also applied.
      game = FactoryGirl.create(:game)
      me = FactoryGirl.create(:person)
      my_teams = FactoryGirl.create_list(:team_with_people, 2).each {|t| t.add_member!(me)}
      other_teams = FactoryGirl.create_list(:team_with_people, 2)

      # Values for `matches` are `[expected, description]`.
      matches = {
        # Expected results:
        new_match(game, me, :rando) => [true, "in which I'm a solo participant"],
        new_match(game, my_teams[0], other_teams[0]) => [true, "in which I'm on a team (1)"],
        new_match(game, my_teams[1], other_teams[1]) => [true, "in which I'm on a team (2)"],
        new_match(game, my_teams[0], my_teams[1]) => [true, "in which I'm on both teams"],
        # Potential false positives:
        new_match(game, :rando, :rando) => [false, "between two randos"],
        new_match(game, other_teams[0], other_teams[1]) => [false, "between two teams I'm not on"],
        new_match(FactoryGirl.create(:game), me, :rando) => [false, "for a different game"],
      }

      get '/v1/matches', params: {user_id: me.user_id, game_id: game.id}
      expect(response.status).to eq(200)

      reply_ids = json_ids
      # Ruby doesn't support destructuring binds in procs (`do |match_id, [exp, desc]|`)? Lame.
      matches.each do |match_id, meta|
        expected, description = meta
        if reply_ids.include?(match_id) != expected
          fail "Match #{description} should#{expected ? "" : " not"} be in reply"
        end
      end
    end
  end

  describe 'PATCH /v1/matches/:id' do
    before :each do
      @match = FactoryGirl.create(:match)
    end

    it 'accepts nested attributes for opponents through opponents' do
      teams = FactoryGirl.create_list(:team, 2)
      patch "/v1/matches/#{@match.id}", params: {
        opponents: @match.opponents.collect{|t| {id: t.id, _destroy: true}} +
          teams.collect{|t| {contender_id: t.id, contender_type: 'Team'}}
      }, headers: admin_headers, as: :json
      expect(response).to be_ok, "not ok: #{json}"
      @match.reload
      expect(@match.team_ids.sort).to eq(teams.collect(&:id).sort)
    end

    it 'requires at least two opponents' do
      patch "/v1/matches/#{@match.id}", params: {
        opponents: @match.opponents.collect{|t| {id: t.id, _destroy: true}}
      }, headers: admin_headers, as: :json
      expect(response).to_not be_ok
      expect(json['errors']).to include('opponents')
    end

    context 'for winner reporting' do
      it 'sets and locks the match winner by admin request' do
        winner_id = @match.opponents[0].id
        patch "/v1/matches/#{@match.id}",
          params: {winner_id: winner_id},
          headers: admin_headers
        expect(response.status).to eq(200)
        expect(json['winner_id']).to eq(winner_id)
        expect(json['state']).to eq('locked')
      end

      it 'allows team captains/owners to update match winner' do
        ['captain', 'owner'].each do |role|
          match = FactoryGirl.create(:match)
          person = FactoryGirl.create(:person)
          match.opponents[0].contender.add_member!(person, [role])
          winner_id = match.opponents[0].id

          patch "/v1/matches/#{match.id}",
            headers: {'User-ID' => person.user_id},
            params: {winner_id: winner_id}
          expect(response.status).to eq(200)
          expect(json['winner_id']).to eq(winner_id)
          winner = json['opponents'].detect {|opp| opp['id'] == winner_id}
          expect(winner&.dig('winner_id')).to eq(winner_id)
        end
      end

      it 'does not allow a non-captain/non-owner to update match winner' do
        person = FactoryGirl.create(:person)
        @match.opponents[0].contender.add_member!(person, ['scrublord'])
        winner_id = @match.opponents[0].id

        patch "/v1/matches/#{@match.id}",
          headers: {'User-ID' => person.user_id},
          params: {winner_id: winner_id}
        expect(response.status).to eq(403)

        @match.reload
        expect(@match.winner_id).to eq(nil)
        expect(@match.state).to eq('')
        @match.opponents.each do |opponent|
          expect(opponent.winner_id).to eq(nil)
        end
      end
    end
  end
end
